package com.monmonkeygroup.openapi.client;

import com.monmonkeygroup.openapi.OpenApiException;
import com.monmonkeygroup.openapi.protocol.ActionName;
import com.monmonkeygroup.openapi.protocol.ActionType;
import com.monmonkeygroup.openapi.protocol.Message;
import com.monmonkeygroup.openapi.protocol.Packet;
import com.monmonkeygroup.openapi.protocol.v1.ProtocolV1;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

public class Client implements ISocketServiceListener, AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(Client.class);
    private final Profile profile;
    private final URI uri;
    private final IClientConn conn;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final AtomicBoolean autoReconnect = new AtomicBoolean(true);
    private final AtomicLong nextPacketTxId = new AtomicLong(1);
    private final Map<Long, CompletableFuture<Packet>> pendingRequests = new ConcurrentHashMap<>();
    private final Map<String, Set<PushHandler>> pushHandlers = new ConcurrentHashMap<>();
    private Timer timer;
    private Timer heartbeatTimer;
    private long heartbeatInterval = 10000;
    private long retryInterval = 1000;
    private int alreadyRetry = 0;
    private int maxRetry = 0;
    private AfterConnected afterConnected;
    private boolean firstConnect = true;
    private int requestTimeout = 15000;

    public Client(Profile profile) {
        try {
            this.uri = new URI(profile.getUri());
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
        this.profile = profile;
        this.conn = new WebsocketConn(this.uri, new ProtocolV1());
        this.conn.addListener(this);
    }

    public void init() throws OpenApiException {
        this.conn.init();
        this.auth();
        this.heartbeatTimer = new Timer();
        this.heartbeatTimer.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                heartbeat();
            }

        }, heartbeatInterval, heartbeatInterval);
    }

    public <T> T doRequest(String actionName, Object request, Class<T> replyClazz) throws OpenApiException {
        if (!conn.isOpen()) {
            throw new OpenApiException(0, "Request rejected: Client not connected");
        }

        Packet requestPacket = Packet.CreateRequest(actionName, nextPacketTxId.getAndIncrement(), request);
        Packet replyPacket = syncDo(requestPacket);
        if (replyPacket.getErrCode() != 0) {
            throw new OpenApiException(replyPacket.getErrCode(), "");
        }
        return replyPacket.deserialize(replyClazz);
    }

    private Packet syncDo(Packet requestPacket) throws OpenApiException {

        Packet replyPacket;
        try {
            sendPacket(requestPacket);
            CompletableFuture<Packet> future = new CompletableFuture<>();
            pendingRequests.put(requestPacket.getTxId(), future);
            replyPacket = future.get(requestTimeout, TimeUnit.MILLISECONDS);
        } catch (TimeoutException e) {
            throw new OpenApiException(0, "Reply timeout");
        } catch (ExecutionException e) {
            if (e.getCause() instanceof OpenApiException) {
                throw (OpenApiException) e.getCause();
            }
            log.error(e.getMessage(), e);
            throw new OpenApiException(0, e.toString());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new OpenApiException(0, e.toString());
        } finally {
            pendingRequests.remove(requestPacket.getTxId());
        }

        return replyPacket;
    }

    public void sendPacket(Packet packet) {
        conn.sendPacket(packet);
    }

    public void addPushHandler(String actionName, PushHandler handler) {
        Set<PushHandler> set = pushHandlers.computeIfAbsent(actionName, k -> ConcurrentHashMap.newKeySet());
        set.add(handler);
    }

    private void heartbeat() {
        if (!this.conn.isOpen()) {
            return;
        }
        Message.HeartbeatRequest request = new Message.HeartbeatRequest(System.currentTimeMillis());
        try {
            doRequest(ActionName.HEARTBEAT_REQUEST, request, Message.HeartbeatReply.class);
        } catch (OpenApiException ignored) {
        }
    }

    private void auth() throws OpenApiException {
        Message.LoginRequest request = new Message.LoginRequest(profile.getUser(), profile.getToken());
        doRequest(ActionName.LOGIN, request, Message.LoginReply.class);
    }

    @Override
    public void onConnected() {
        if (firstConnect) {
            firstConnect = false;
            return;
        }
        new Thread(() -> {
            try {
                this.auth();
                if (null != afterConnected) {
                    afterConnected.onAfterConnected(this);
                }
            } catch (OpenApiException e) {
                log.error(e.getMessage());
                conn.close("Login failed");
            }
        }).start();
    }

    @Override
    public void onMessage(Packet packet) {
        if (Objects.equals(packet.getActionType(), ActionType.UPDATE)) {
            handlePush(packet);
        } else if (Objects.equals(packet.getActionType(), ActionType.REPLY)) {
            handleReply(packet);
        }
    }

    private void handlePush(Packet packet) {
        Set<PushHandler> handlers = pushHandlers.get(packet.getActionName());
        if (null == handlers) return;

        handlers.forEach(handler -> handler.handlePush(packet));
    }

    private void handleReply(Packet packet) {
        CompletableFuture<Packet> future = pendingRequests.get(packet.getTxId());
        if (null == future) {
            log.warn("no reply handler for txId: {}", packet.getTxId());
            return;
        }

        if (future.isCancelled() || future.isDone()) {
            log.warn("duplicate reply of txId: {}", packet.getTxId());
            return;
        }

        future.complete(packet);
    }

    @Override
    public void onClose(String reason) {
        for (CompletableFuture<Packet> future : this.pendingRequests.values()) {
            future.completeExceptionally(new OpenApiException(0, "Request failed due to connection loss"));
        }

        if (!running.get() || !autoReconnect.get()) {
            return;
        }

        log.debug("reconnect for conn closed: {}", reason);
        this.startTimer();
    }

    private void reconnect() {
        try {
            this.conn.reconnectSocket();
            alreadyRetry = 0;
            this.stopTimer();
            return;
        } catch (Exception ignored) {
        }

        alreadyRetry++;
        if (maxRetry != 0 && alreadyRetry >= maxRetry) {
            stopTimer();
        }
    }

    public synchronized void startTimer() {
        if (null != this.timer) {
            return;
        }
        this.timer = new Timer("SocketReconnectingThread-" + System.currentTimeMillis());
        this.timer.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                reconnect();
            }

        }, 0, retryInterval);
    }

    public synchronized void stopTimer() {
        this.timer.cancel();
        this.timer = null;
    }

    @Override
    public synchronized void close() {
        if (!running.get()) {
            return;
        }

        running.set(false);
        autoReconnect.set(false);

        heartbeatTimer.cancel();
        if (null != conn) {
            conn.close("close by client");
        }

        if (null != timer) {
            stopTimer();
        }
    }

    public void setHeartbeatInterval(long heartbeatInterval) {
        this.heartbeatInterval = heartbeatInterval;
    }

    public void setRetryInterval(long retryInterval) {
        this.retryInterval = retryInterval;
    }

    public void setMaxRetry(int maxRetry) {
        this.maxRetry = maxRetry;
    }

    public void setRequestTimeout(int requestTimeout) {
        this.requestTimeout = requestTimeout;
    }

    public void setAfterConnected(AfterConnected afterConnected) {
        this.afterConnected = afterConnected;
    }

    public interface AfterConnected {
        void onAfterConnected(Client client);
    }
}
