package io.github.kaststream.api.v2.facade;

import java.util.Set;
import javax.annotation.Nullable;

public interface KastRow {

  /**
   * Returns the kind of change that this row describes in a changelog.
   *
   * <p>By default, a row describes an {@link KastRowKind#INSERT} change.
   *
   * @see KastRowKind
   */
  public KastRowKind getKind();

  /**
   * Sets the kind of change that this row describes in a changelog.
   *
   * <p>By default, a row describes an {@link KastRowKind#INSERT} change.
   *
   * @see KastRowKind
   */
  public void setKind(KastRowKind kind);

  /**
   * Returns the number of fields in the row.
   *
   * <p>Note: The row kind is kept separate from the fields and is not included in this number.
   *
   * @return the number of fields in the row
   */
  public int getArity();

  /**
   * Returns the field's content at the specified field position.
   *
   * <p>Note: The row must operate in position-based field mode.
   *
   * @param pos the position of the field, 0-based
   * @return the field's content at the specified position
   */
  public @Nullable Object getField(int pos);

  /**
   * Returns the field's content at the specified field position.
   *
   * <p>Note: The row must operate in position-based field mode.
   *
   * <p>This method avoids a lot of manual casting in the user implementation.
   *
   * @param pos the position of the field, 0-based
   * @return the field's content at the specified position
   */
  public <T> T getFieldAs(int pos);

  /**
   * Returns the field's content using the specified field name.
   *
   * <p>Note: The row must operate in name-based field mode.
   *
   * @param name the name of the field or null if not set previously
   * @return the field's content
   */
  public @Nullable Object getField(String name);

  /**
   * Returns the field's content using the specified field name.
   *
   * <p>Note: The row must operate in name-based field mode.
   *
   * <p>This method avoids a lot of manual casting in the user implementation.
   *
   * @param name the name of the field, set previously
   * @return the field's content
   */
  public <T> T getFieldAs(String name);

  /**
   * Sets the field's content at the specified position.
   *
   * <p>Note: The row must operate in position-based field mode.
   *
   * @param pos the position of the field, 0-based
   * @param value the value to be assigned to the field at the specified position
   */
  public void setField(int pos, @Nullable Object value);

  /**
   * Sets the field's content using the specified field name.
   *
   * <p>Note: The row must operate in name-based field mode.
   *
   * @param name the name of the field
   * @param value the value to be assigned to the field
   */
  public void setField(String name, @Nullable Object value);

  /**
   * Returns the set of field names if this row operates in name-based field mode, otherwise null.
   *
   * <p>This method is a helper method for serializers and converters but can also be useful for
   * other row transformations.
   *
   * @param includeNamedPositions whether or not to include named positions when this row operates
   *     in a hybrid field mode
   */
  public @Nullable Set<String> getFieldNames(boolean includeNamedPositions);

  /** Clears all fields of this row. */
  public void clear();
}
