package io.github.kaststream.api.v2.facade;

public interface KastRowBuilder {

  /**
   * Creates a fixed-length row in position-based field mode.
   *
   * <p>The semantics are equivalent to {@link
   * KastRowBuilder#createKastRowWithPositions(KastRowKind, int)}. This constructor exists for
   * backwards compatibility.
   *
   * @param kind kind of change a row describes in a changelog
   * @param arity the number of fields in the row
   */
  KastRow createKastRow(KastRowKind kind, int arity);

  /**
   * Creates a fixed-length row in position-based field mode.
   *
   * <p>The semantics are equivalent to {@link KastRowBuilder#createKastRowWithPositions(int)}. This
   * constructor exists for backwards compatibility.
   *
   * @param arity the number of fields in the row
   */
  KastRow createKastRow(int arity);

  /**
   * Creates a fixed-length row in position-based field mode.
   *
   * <p>Fields can be accessed by position via {@link KastRow#setField(int, Object)} and {@link
   * KastRow#getField(int)}.
   *
   * <p>See the class documentation of {@link KastRow} for more information.
   *
   * @param kind kind of change a row describes in a changelog
   * @param arity the number of fields in the row
   * @return a new row instance
   */
  KastRow createKastRowWithPositions(KastRowKind kind, int arity);

  /**
   * Creates a fixed-length row in position-based field mode.
   *
   * <p>Fields can be accessed by position via {@link KastRow#setField(int, Object)} and {@link
   * KastRow#getField(int)}.
   *
   * <p>By default, a row describes an {@link KastRowKind#INSERT} change.
   *
   * <p>See the class documentation of {@link KastRow} for more information.
   *
   * @param arity the number of fields in the row
   * @return a new row instance
   */
  KastRow createKastRowWithPositions(int arity);

  /**
   * Creates a variable-length row in name-based field mode.
   *
   * <p>Fields can be accessed by name via {@link KastRow#setField(String, Object)} and {@link
   * KastRow#getField(String)}.
   *
   * <p>See the class documentation of {@link KastRowKind} for more information.
   *
   * @param kind kind of change a row describes in a changelog
   * @return a new row instance
   */
  KastRow createKastRowWithNames(KastRowKind kind);

  /**
   * Creates a variable-length row in name-based field mode.
   *
   * <p>Fields can be accessed by name via {@link KastRow#setField(String, Object)} and {@link
   * KastRow#getField(String)}.
   *
   * <p>By default, a row describes an {@link KastRowKind#INSERT} change.
   *
   * <p>See the class documentation of {@link KastRow} for more information.
   *
   * @return a new row instance
   */
  KastRow createKastRowWithNames();

  /**
   * Creates a fixed-length row in position-based field mode and assigns the given values to the
   * row's fields.
   *
   * <p>This method should be more convenient than {@link
   * KastRowBuilder#createKastRowWithPositions(int)} (int)} in many cases.
   *
   * <p>For example:
   *
   * <pre>
   *     kastRowBuilder.createKastRowOf("hello", true, 1L);
   * </pre>
   *
   * instead of
   *
   * <pre>
   *     KastRow row = kastRowBuilder.createKastRowWithPositions(3);
   *     row.setField(0, "hello");
   *     row.setField(1, true);
   *     row.setField(2, 1L);
   * </pre>
   *
   * <p>By default, a row describes an {@link KastRowKind#INSERT} change.
   */
  KastRow createKastRowOf(Object... values);

  /**
   * Creates a fixed-length row in position-based field mode with given kind and assigns the given
   * values to the row's fields.
   *
   * <p>This method should be more convenient than {@link
   * KastRowBuilder#createKastRowWithPositions(KastRowKind, int)} (RowKind, int)} in many cases.
   *
   * <p>For example:
   *
   * <pre>
   *     kastRowBuilder.createKastRowOfKind(RowKind.INSERT, "hello", true, 1L);
   * </pre>
   *
   * instead of
   *
   * <pre>
   *     KastRow row = kastRowBuilder.createKastRowWithPositions(3);
   *     row.setField(0, "hello");
   *     row.setField(1, true);
   *     row.setField(2, 1L);
   * </pre>
   */
  KastRow createKastRowOfKind(KastRowKind kind, Object... values);

  /**
   * Creates a new row which is copied from another row (including its {@link KastRowKind}).
   *
   * <p>This method does not perform a deep copy.
   */
  KastRow kastRowCopy(KastRow kastRow);

  /**
   * Creates a new row with projected fields and identical {@link KastRowKind} from another row.
   *
   * <p>This method does not perform a deep copy.
   *
   * <p>Note: The row must operate in position-based field mode. Field names are not projected.
   *
   * @param fieldPositions field indices to be projected
   */
  KastRow kastRowProject(KastRow kastRow, int[] fieldPositions);

  /**
   * Creates a new row with projected fields and identical {@link KastRowKind} from another row.
   *
   * <p>This method does not perform a deep copy.
   *
   * <p>Note: The row must operate in name-based field mode.
   *
   * @param fieldNames field names to be projected
   */
  KastRow kastRowProject(KastRow row, String[] fieldNames);

  /**
   * Creates a new row with fields that are copied from the other rows and appended to the resulting
   * row in the given order. The {@link KastRowKind} of the first row determines the {@link
   * KastRowKind} of the result.
   *
   * <p>This method does not perform a deep copy.
   *
   * <p>Note: All rows must operate in position-based field mode.
   */
  KastRow kastRowJoin(KastRow first, KastRow... remainings);
}
