/*
 *                    BioJava development code
 *
 * This code may be freely distributed and modified under the
 * terms of the GNU Lesser General Public Licence.  This should
 * be distributed with the code.  If you do not have a copy,
 * see:
 *
 *      http://www.gnu.org/copyleft/lesser.html
 *
 * Copyright for this code is held jointly by the individual
 * authors.  These should be listed in @author doc comments.
 *
 * For more information on the BioJava project and its aims,
 * or to join the biojava-l mailing list, visit the home page
 * at:
 *
 *      http://www.biojava.org/
 *
 * Created on DATE
 *
 */
package org.biojava.nbio.core.sequence;


import org.biojava.nbio.core.exceptions.CompoundNotFoundException;
import org.biojava.nbio.core.sequence.compound.DNACompoundSet;
import org.biojava.nbio.core.sequence.compound.NucleotideCompound;
import org.biojava.nbio.core.sequence.template.CompoundSet;

/**
 * Represents a exon or coding sequence in a gene. It has a parent {@link TranscriptSequence}
 * where a TranscriptSequence is the child of a GeneSequence
 * Not important for protein construction but the phase is used if outputting the gene
 * to a gff3 file. <a href="http://www.sequenceontology.org/gff3.shtml">http://www.sequenceontology.org/gff3.shtml</a>
 * @author Scooter Willis
 */
public class CDSSequence extends DNASequence {

	//private static final Logger log = Logger.getLogger(CDSSequence.class.getName());
	Integer phase = 0; // 0, 1, 2
	TranscriptSequence parentTranscriptSequence;

	/**
	 *
	 * @param parentSequence
	 * @param bioBegin
	 * @param bioEnd
	 * @param phase
	 * @throws  IllegalArgumentException if  parentSequence is incompatible with DNACompoundSet
	 */
	public CDSSequence(TranscriptSequence parentSequence, int bioBegin, int bioEnd, int phase) {
		setCompoundSet(DNACompoundSet.getDNACompoundSet());
		try {
			initSequenceStorage(parentSequence.getSequenceAsString());
		} catch (CompoundNotFoundException e) {
			throw new IllegalArgumentException(e);
		}
		parentTranscriptSequence = parentSequence;
		this.setParentSequence(parentTranscriptSequence);
		setBioBegin(bioBegin);
		setBioEnd(bioEnd);
		this.phase = phase;
		this.setCompoundSet(DNACompoundSet.getDNACompoundSet());

	}

		@Override
	public int getLength() {
		return Math.abs(this.getBioEnd() - this.getBioBegin()) + 1;
	}

	/**
	 *
	 * @return get the phase
	 */
	public Integer getPhase() {
		return phase;
	}

	/**
	 *
	 * @return get the strand
	 */
	public Strand getStrand() {
		return parentTranscriptSequence.getStrand();
	}

	/**
	 * A CDS sequence if negative stranded needs to be reverse complement
	 * to represent the actual coding sequence. When getting a ProteinSequence
	 * from a TranscriptSequence this method is callled for each CDSSequence
	 * <a href="http://www.sequenceontology.org/gff3.shtml">http://www.sequenceontology.org/gff3.shtml</a>
	 * <a href="http://biowiki.org/~yam/bioe131/GFF.ppt">http://biowiki.org/~yam/bioe131/GFF.ppt</a>
	 * @return coding sequence
	 */
	public String getCodingSequence() {
		String sequence = this.getSequenceAsString(getBioBegin(), getBioEnd(), getStrand());

		if (getStrand() == Strand.NEGATIVE) {
			//need to take complement of sequence because it is negative and we are returning a coding sequence
			StringBuilder b = new StringBuilder(getLength());
			CompoundSet<NucleotideCompound> compoundSet = this.getCompoundSet();
			for (int i = 0; i < sequence.length(); i++) {
				String nucleotide = String.valueOf(sequence.charAt(i));
				NucleotideCompound nucleotideCompound = compoundSet.getCompoundForString(nucleotide);
				b.append(nucleotideCompound.getComplement().getShortName());
			}
			sequence = b.toString();
		}
		//  sequence = sequence.substring(phase);
		return sequence;
	}
}
