package org.quickjava.orm.loader;

import cn.hutool.core.util.ObjectUtil;
import org.quickjava.orm.contain.DatabaseConfig;
import org.quickjava.orm.model.Model;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnSingleCandidate;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;

@Configuration
@ConditionalOnSingleCandidate(DataSource.class)
public class SpringLoader implements InitializingBean {

    private static final Logger logger = LoggerFactory.getLogger(SpringLoader.class);

    private static ApplicationContext applicationContext;

    public static SpringLoader instance;

//    public static MybatisProperties mybatisProperties;

//    public static MybatisPlusProperties mybatisPlusProperties;

    @Autowired
    private Environment environment;

    @Autowired
    private DataSource druidDataSource;

    private DatabaseConfig.DBType dbType;

    private DatabaseConfig dbConfig;

    public SpringLoader(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
        this.instance = this;
    }

    public static ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.loadModel();
    }

    public DataSource getDataSource() {
        return druidDataSource;
    }

    public DatabaseConfig.DBType getConnectionType() {
        synchronized (SpringLoader.class) {
            if (dbType == null) {
                try {
                    Connection connection = getDataSource().getConnection();
                    dbType = DatabaseConfig.parseTypeFromConnection(connection);
                    connection.close();
                } catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        return dbType;
    }

    public DatabaseConfig getConfig() {
        if (dbConfig == null) {
            dbConfig = new DatabaseConfig(DatabaseConfig.DBSubject.SPRING, getConnectionType());
        }
        return dbConfig;
    }

    public Environment getEnvironment() {
        return environment;
    }

    private void loadModel() {
        String typeAliasesPackage = environment.getProperty("mybatis.type-aliases-package");
        if (ObjectUtil.isEmpty(typeAliasesPackage)) {
            typeAliasesPackage = environment.getProperty("mybatis-plus.type-aliases-package");
        }
        if (ObjectUtil.isEmpty(typeAliasesPackage)) {
            typeAliasesPackage = environment.getProperty("spring.component-scan.base-package");
        }
        logger.debug("Load model entities, in package: " + typeAliasesPackage);
        if (typeAliasesPackage == null) {
            logger.warn("model entities package path error");
            return;
        }

        ClassPathScanningCandidateComponentProvider scanner2 = new ClassPathScanningCandidateComponentProvider(false);
        scanner2.addIncludeFilter((m, f) -> {
            try {
                Class<?> clazz = Class.forName(m.getClassMetadata().getClassName());
                if (Model.class.isAssignableFrom(clazz)) {
                    logger.debug("Load model entity: {}", clazz.getName());
                    clazz.newInstance();
                }
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (InstantiationException | IllegalAccessException e) {
                throw new RuntimeException(e);
            }
            return false;
        });
        // 扫描指定包
        scanner2.findCandidateComponents(typeAliasesPackage);
    }
}
