package com.flutterwave.services;

import com.flutterwave.bean.ListResponse;
import com.flutterwave.bean.Response;
import com.flutterwave.client.Client;
import org.apache.http.NameValuePair;
import org.apache.http.message.BasicNameValuePair;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static com.flutterwave.bean.ChargeTypes.SUBSCRIPTION;
import static com.flutterwave.bean.Verb.GET;
import static com.flutterwave.client.Utility.put;
import static com.flutterwave.utility.Properties.getProperty;

/**
 * API endpoints that are critical for managing user subscriptions.
 * With these APIs, You can query your existing subscriptions, as well as activate or deactivate a user's subscription.
 * @author Cleopatra Douglas
 */
public class Subscriptions {
    private String ERROR = "Error processing request, please check logs";

    /**
     * Query all subscribers (cancelled subscriber included). You can do a single or bulk query with the endpoint depending on your use case.
     * @param email Optional This is the email of the subscriber. You can use this to query the transactions for a single customer.
     * @param transaction_id Optional This is a unique transaction identifier generated by our systems. It is returned in the initiate charge response as data.id.
     * @param plan Optional This is the ID of the payment plan. It is returned in the call to create a payment plan as data.id.
     * @param subscribed_from Optional This is the start date of the subscriptions(when they are first active). You can use this as a starting point for your query. The expected date format for this parameter is YYYY-MM-DD
     * @param subscribed_to Optional This is the end date for a subscription. Like subscribed_from, It also has an expected date format of YYYY-MM-DD.
     * @param next_due_from Optional This is the parameter to filter from the start date of the next due subscriptions.
     * @param next_due_to Optional This is the parameter to filter to the end date of the next due subscriptions.
     * @param page Optional This is the page number to retrieve e.g. setting 1 retrieves the first page.
     * @param status Optional This is the status for the queried transactions. Expected values are active and cancelled.
     * @return ListResponse
     */
    public ListResponse getAllSubscriptions(Optional<String> email, Optional<Integer> transaction_id, Optional<Integer> plan,
                                            Optional<String> subscribed_from,  Optional<String> subscribed_to, Optional<String> next_due_from,
                                            Optional<String> next_due_to,  Optional<String> page, Optional<String> status){
        List<NameValuePair> nameValuePairs = new ArrayList<>();
        email.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("email", s)));
        transaction_id.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("transaction_id", s.toString())));
        plan.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("plan", s.toString())));
        subscribed_from.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("subscribed_from", s)));
        subscribed_to.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("subscribed_to", s)));
        next_due_from.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("next_due_from", s)));
        next_due_to.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("next_due_to", s)));
        page.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("page", s)));
        status.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("status", s)));

        return Optional.ofNullable(Client.runTransaction(
                        getProperty("SUBSCRIPTION_BASE"),
                        null,
                        GET,
                        SUBSCRIPTION,
                        nameValuePairs))
                .map(ListResponse::toListResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    /**
     * Activate a previously cancelled subscription.
     * @param id int This is the ID of the subscription you want to activate. It is returned in the call to query all subscription as data.id.
     * @return Response
     */
    public Response activateSubscription(int id){
        return Optional.of(put(getProperty("SUBSCRIPTION_BASE") + "/" + id + "/activate",
                        null, SUBSCRIPTION, null))
                .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    /**
     * Deactivate an active subscription.
     * @param id int This is the ID of the subscription to be cancelled. It is returned in the call to query all subscription as data.id.
     * @return Response
     */
    public Response deactivateSubscription(int id) {

        return Optional.of(put(getProperty("SUBSCRIPTION_BASE") + "/" + id + "/cancel",
                        null, SUBSCRIPTION, null))
                .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }
}
