package com.github.collinalpert.java2db.database;

import com.github.collinalpert.java2db.exceptions.ConnectionFailedException;
import com.github.collinalpert.java2db.mappers.FieldMapper;
import com.github.collinalpert.java2db.queries.*;
import com.github.collinalpert.java2db.queries.async.*;
import com.mysql.cj.exceptions.CJCommunicationsException;
import com.mysql.cj.jdbc.exceptions.CommunicationsException;

import java.io.Closeable;
import java.sql.*;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import static com.github.collinalpert.java2db.utilities.Utilities.supplierHandling;

/**
 * @author Collin Alpert
 */
public class DBConnection implements Closeable {

	/**
	 * Specifies the hostname/ip address of the database.
	 */
	public static String HOST;

	/**
	 * Specifies the name of the database to connect to.
	 */
	public static String DATABASE;

	/**
	 * Specifies the username to log in on the database with.
	 */
	public static String USERNAME;

	/**
	 * Specifies the password to log in on the database with.
	 */
	public static String PASSWORD;

	/**
	 * Specifies the port to connect to the database on.
	 * This property is optional. If not specified, it will be set to 3306, the default port of MySQL.
	 */
	public static int PORT = 3306;

	/**
	 * Specifies the login timeout to the database in seconds.
	 */
	public static int TIMEOUT = 5;

	/**
	 * Constant which determines if the queries generated by Java2DB will be logged in the console.
	 */
	public static boolean LOG_QUERIES = true;

	private Connection underlyingConnection;
	private boolean isConnectionValid;

	public DBConnection() {
		try {
			var connectionString = String.format("jdbc:mysql://%s:%d/%s?rewriteBatchedStatements=true", HOST, PORT, DATABASE);
			Class.forName("com.mysql.cj.jdbc.Driver");
			System.setProperty("user", USERNAME);
			System.setProperty("password", PASSWORD);
			DriverManager.setLoginTimeout(TIMEOUT);
			underlyingConnection = DriverManager.getConnection(connectionString, System.getProperties());
			isConnectionValid = true;
		} catch (CJCommunicationsException | CommunicationsException e) {
			isConnectionValid = false;
			throw new ConnectionFailedException();
		} catch (ClassNotFoundException | SQLException e) {
			e.printStackTrace();
			isConnectionValid = false;
		}
	}

	public DBConnection(Connection underlyingConnection) {
		this.underlyingConnection = underlyingConnection;
		this.isConnectionValid = true;
	}

	/**
	 * Checks if the connection is valid/successful.
	 *
	 * @return True if connection was successful, false if not.
	 */
	public boolean isValid() {
		return this.isConnectionValid;
	}


	/**
	 * Executes a DQL statement on the database without Java parameters.
	 *
	 * @param query The query to be executed.
	 * @return The {@link ResultSet} containing the result from the DQL statement.
	 * @throws SQLException if the query is malformed or cannot be executed.
	 */
	public ResultSet execute(String query) throws SQLException {
		var statement = this.underlyingConnection.createStatement();
		log(query);
		var set = statement.executeQuery(query);
		statement.closeOnCompletion();

		return set;
	}

	/**
	 * Executes a DQL statement on the database with Java parameters.
	 *
	 * @param query  The query to be executed.
	 * @param params The Java parameters to be inserted into the query.
	 * @return The {@link ResultSet} containing the result from the DQL statement.
	 * @throws SQLException if the query is malformed or cannot be executed.
	 */
	public ResultSet execute(String query, Object... params) throws SQLException {
		var statement = this.underlyingConnection.prepareStatement(query);
		for (int i = 0; i < params.length; i++) {
			statement.setObject(i + 1, params[i]);
		}

		log(query);
		var set = statement.executeQuery();
		statement.closeOnCompletion();
		return set;
	}

	/**
	 * This command is used for any DDL/DML queries.
	 *
	 * @param query The query to be executed.
	 * @return the last generated ID. This return value should only be used with INSERT statements.
	 * @throws SQLException if the query is malformed or cannot be executed.
	 */
	public long update(String query) throws SQLException {
		var statement = this.underlyingConnection.createStatement();
		log(query);
		statement.executeUpdate(query, Statement.RETURN_GENERATED_KEYS);
		return updateInternal(statement);
	}

	/**
	 * This command is used for any DDL/DML queries with Java parameters.
	 *
	 * @param query  The query to be executed.
	 * @param params The Java parameters to be inserted into the query.
	 * @return the last generated ID. This return value should only be used with INSERT statements.
	 * @throws SQLException if the query is malformed or cannot be executed.
	 */
	public long update(String query, Object... params) throws SQLException {
		var statement = this.underlyingConnection.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
		for (int i = 0; i < params.length; i++) {
			statement.setObject(i + 1, params[i]);
		}

		log(query);
		statement.executeUpdate();
		return updateInternal(statement);
	}

	/**
	 * Determines if a connection to the database still exists or not.
	 *
	 * @return {@code True} if a connection exists, {@code false} if not.
	 * This method will return {@code false} if an exception occurs.
	 */
	public boolean isOpen() {
		try {
			return !this.underlyingConnection.isClosed();
		} catch (SQLException e) {
			System.err.println("Could not determine connection status");
			return this.isConnectionValid = false;
		}
	}

	/**
	 * Closes the connection to the database.
	 */
	@Override
	public void close() {
		try {
			if (this.underlyingConnection != null) {
				this.underlyingConnection.close();
			}
		} catch (SQLException e) {
			System.err.println("Could not close database connection");
			e.printStackTrace();
		} finally {
			this.isConnectionValid = false;
		}
	}

	/**
	 * Calls an SQL function.
	 *
	 * @param returnType   The Java equivalent of the SQL datatype the function returns.
	 * @param functionName The name of the function.
	 * @param arguments    The arguments to be supplied to the function.
	 * @param <T>          The functions return type.
	 * @return The value of the function, as a Java datatype.
	 * @throws SQLException In case there is an error communicating with the database, i.e. the function does not exist.
	 */
	public <T> Optional<T> callFunction(Class<T> returnType, String functionName, Object... arguments) throws SQLException {
		var joiner = new StringJoiner(",");
		for (int i = 0; i < arguments.length; i++) {
			joiner.add("?");
		}

		try (var set = execute(String.format("select %s(%s);", functionName, joiner.toString()), arguments)) {
			return new FieldMapper<>(returnType).map(set);
		}
	}

	public <T> CompletableFuture<Optional<T>> callFunctionAsync(Consumer<SQLException> exceptionHandler, Class<T> returnType, String functionName, Object... arguments) {
		return CompletableFuture.supplyAsync(supplierHandling(() -> this.callFunction(returnType, functionName, arguments), exceptionHandler));
	}

	public <T> CompletableFuture<Void> callFunctionAsync(Consumer<SQLException> exceptionHandler, Consumer<? super Optional<T>> callback, Class<T> returnType, String functionName, Object... arguments) {
		return CompletableFuture.supplyAsync(supplierHandling(() -> this.callFunction(returnType, functionName, arguments), exceptionHandler)).thenAcceptAsync(callback);
	}

	public <T> Queryable<T> callStoredProcedure(Class<T> returnType, String storedProcedureName, Object... arguments) {
		return new StoredProcedureQuery<>(returnType, this, storedProcedureName, arguments);
	}

	public <T> AsyncQueryable<T> callStoredProcedureAsync(Class<T> returnType, String storedProcedureName, Object... arguments) {
		return new AsyncStoredProcedureQuery<>(returnType, this, storedProcedureName, arguments);
	}

	public Connection underlyingConnection() {
		return this.underlyingConnection;
	}

	/**
	 * Prints queries to the console, while considering the {@link DBConnection#LOG_QUERIES} constant.
	 *
	 * @param text The message to print.
	 */
	private void log(String text) {
		if (LOG_QUERIES) {
			System.out.println(text);
		}
	}

	private long updateInternal(Statement statement) throws SQLException {
		statement.closeOnCompletion();
		var set = statement.getGeneratedKeys();
		if (set.next()) {
			return set.getLong(1);
		}

		return -1;
	}
}
