/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.apithefire.sql.connect;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apithefire.sql.RuntimeSqlException;
import org.apithefire.util.lang.Objects;

/**
 * A {@link DataSource} wrapper to provide {@link ConnectionProvider} interface.
 * 
 * @author <a href="mailto:bachew@gmail.com">Chew Boon Aik</a>
 */
// TODO: Class name is confusing
public class ConnectionProviderWrapper implements DataSource {
	
	private final PrintWriter nullPrintWriter = newNullPrintWriter();
	private final ConnectionProvider connectionProvider;
	
	/**
	 * Wraps the specified connection provider.
	 * 
	 * @return data source wrapper. If the specified connection provider is
	 *         instance of {@link DataSourceWrapper}, the wrapped data source
	 *         is returned.
	 */
	public static DataSource wrap(ConnectionProvider connectionProvider) {
		if (connectionProvider instanceof DataSourceWrapper) {
			return ((DataSourceWrapper) connectionProvider)
					.getWrappedDataSource();
		}
		return new ConnectionProviderWrapper(connectionProvider);
	}
	
	private ConnectionProviderWrapper(ConnectionProvider connectionProvider) {
		this.connectionProvider = Objects.nonNull(connectionProvider);
	}

	public Connection getConnection() throws SQLException {
		try {
			return connectionProvider.getConnection();
		} catch (RuntimeSqlException e) {
			Throwable cause = e.getCause();
			
			if (cause != null && cause instanceof SQLException) {
				throw (SQLException) cause;
			}
			throw e;
		}
	}

	public Connection getConnection(String username, String password)
			throws SQLException {
		return getConnection();
	}
	
	private static PrintWriter newNullPrintWriter() {
		return new PrintWriter(new Writer() {
			@Override
			public void close() throws IOException {
			}
			@Override
			public void flush() throws IOException {
			}
			@Override
			public void write(char[] arg0, int arg1, int arg2)
					throws IOException {
			}
		});
	}

	public PrintWriter getLogWriter() throws SQLException {
		return nullPrintWriter;
	}

	public int getLoginTimeout() throws SQLException {
		return 0;
	}

	public void setLoginTimeout(int seconds) throws SQLException {
	}

	public void setLogWriter(PrintWriter out) throws SQLException {
	}

	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		return false;
	}

	public <T> T unwrap(Class<T> iface) throws SQLException {
		throw new SQLException("Can not unwrap " + iface);
	}

}
