/*
 * Copyright (C) ${project.inceptionYear} Benny Bottema (benny@bennybottema.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.bbottema.javareflection;

import org.bbottema.javareflection.model.InvokableObject;
import org.bbottema.javareflection.model.LookupMode;
import org.bbottema.javareflection.util.ArrayKey;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * For internal use for improving repeated lookup performances.
 */
public class LookupCaches {
	
	/**
	 * {@link Class} cache optionally used when looking up classes with {@link ClassUtils#locateClass(String, boolean, ClassLoader)}.
	 */
	final static Map<String, Class<?>> CLASS_CACHE = new HashMap<>();
	
	/**
	 * {@link Method} cache categorized by owning <code>Classes</code> (since several owners can have a method with the same name and signature).
	 * Methods are stored based on <code>Method</code> reference along with their unique signature (per owner), so multiple methods on one owner with
	 * the same name can coexist.<br />
	 * <br />
	 * This cache is being maintained to reduce lookup times when trying to find signature compatible Java methods. The possible signature
	 * combinations using autoboxing and/or automatic common conversions can become very large (7000+ with only three parameters) and can become a
	 * real problem. The more frequently a method is being called the larger the performance gain, especially for methods with long parameter lists
	 *
	 * @see "MethodUtils.addMethodToCache(Class, String, Set, Class[])"
	 * @see "MethodUtils#getMethodFromCache(Class, String, Class[])"
	 */
	final static Map<Class<?>, Map<String, Map<Class<?>[], Set<InvokableObject>>>> METHOD_CACHE = new LinkedHashMap<>();
	
	static final Map<Class<?>, Set<Class<?>>> CACHED_REGISTERED_COMPATIBLE_TARGET_TYPES = new HashMap<>();
	static final Map<Class<?>, Set<Class<?>>> CACHED_COMPATIBLE_TARGET_TYPES = new HashMap<>();
	private static final Map<Set<LookupMode>, Map<ArrayKey, List<Class<?>[]>>> CACHED_COMPATIBLE_TYPE_LISTS = new HashMap<>();
	
	@SuppressWarnings({"unused"})
	public static void resetCache() {
		CLASS_CACHE.clear();
		METHOD_CACHE.clear();
		CACHED_REGISTERED_COMPATIBLE_TARGET_TYPES.clear();
		CACHED_COMPATIBLE_TARGET_TYPES.clear();
		CACHED_COMPATIBLE_TYPE_LISTS.clear();
	}
	
	@Nullable
	static List<Class<?>[]> getCachedCompatibleSignatures(Set<LookupMode> lookupMode, ArrayKey arrayKey) {
		final Map<ArrayKey, List<Class<?>[]>> cachedCompatibleSignatures = CACHED_COMPATIBLE_TYPE_LISTS.get(lookupMode);
		if (cachedCompatibleSignatures != null) {
			return cachedCompatibleSignatures.get(arrayKey);
		}
		return null;
	}
	
	@NotNull
	static List<Class<?>[]> addCompatiblesignaturesToCache(Set<LookupMode> lookupMode, ArrayKey arrayKey, List<Class<?>[]> compatibleTypeLists) {
		Map<ArrayKey, List<Class<?>[]>> cachedCompatibleSignatures = CACHED_COMPATIBLE_TYPE_LISTS.get(lookupMode);
		if (cachedCompatibleSignatures == null) {
			CACHED_COMPATIBLE_TYPE_LISTS.put(lookupMode, cachedCompatibleSignatures = new HashMap<>());
		}
		cachedCompatibleSignatures.put(arrayKey, compatibleTypeLists);
		return compatibleTypeLists;
	}
}