package cdc.kernel.validation;

import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.jupiter.api.Test;

import cdc.converters.defaults.StringToInteger;
import cdc.util.lang.Introspection;
import cdc.validation.checkers.Checker;
import cdc.validation.checkers.defaults.IsInRange;
import cdc.validation.checkers.defaults.MatchesPattern;

class CheckerTest {
    private static final Logger LOGGER = LogManager.getLogger(CheckerTest.class);

    @Test
    void testNegate() {
        final Checker<Integer> checker = IsInRange.from(0, 100).negate();
        assertTrue(checker.test(200));
        LOGGER.debug(checker.explain());
    }

    @Test
    void testAnd() {
        final Checker<Integer> checker = IsInRange.from(0, 100).and(IsInRange.from(50, 150));
        assertTrue(checker.test(75));
        assertFalse(checker.test(200));
        LOGGER.debug(checker.explain());
    }

    @Test
    void testOr() {
        final Checker<Integer> checker = IsInRange.from(0, 100).or(IsInRange.from(50, 150));
        assertTrue(checker.test(75));
        assertFalse(checker.test(200));
        LOGGER.debug(checker.explain());
    }

    @Test
    void testAfter() {
        final Checker<String> checker = IsInRange.from(10, 20).after(StringToInteger.INSTANCE);
        assertTrue(checker.test("15"));
        assertFalse(checker.test("30"));
        assertFalse(checker.test("Hello"));
        LOGGER.debug(checker.explain());
    }

    @Test
    void testAndCombining() {
        final Checker<Integer> checker =
                Checker.and(IsInRange.from(10, 20),
                            IsInRange.from(15, 25));
        assertTrue(checker.test(15));
        assertFalse(checker.test(10));
        LOGGER.debug(checker.testAndExplain(0));
    }

    @Test
    void testOrCombining() {
        final Checker<Integer> checker =
                Checker.or(IsInRange.from(10, 20),
                           IsInRange.from(15, 25));
        assertTrue(checker.test(15));
        assertTrue(checker.test(10));
        assertFalse(checker.test(100));
    }

    @Test
    void testCollectionChecker() {
        final Checker<List<String>> checker =
                Checker.from(Introspection.uncheckedCast(List.class),
                             MatchesPattern.DIGITS);
        assertTrue(checker.test(List.of()));
        assertTrue(checker.test(List.of("000")));
        assertFalse(checker.test(List.of("A")));
        assertFalse(checker.test(List.of("0", "A", "1")));
        LOGGER.debug(checker.testAndExplain(List.of("000")));
        LOGGER.debug(checker.testAndExplain(List.of("A")));
    }
}