package fi.evolver.script.app;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;
import java.util.regex.Pattern;

import fi.evolver.script.Dialog;
import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class Npm {
	private static final Duration VALIDATION_TIMEOUT = Duration.ofSeconds(10);

	public static void setupBashCompletion() {
		try (Step step = Step.start("Setup NPM bash completion")) {
			FileUtils.writeShellBlock(Shell.BASHRC, "NPM bash completion", """
					source <(npm completion)
					""");
		}
	}

	public static void setupCredentials(Repo repo) {
		try (Step step = Step.start("Setup NPM credentials")) {
			Path npmrc = Shell.HOME.resolve(".npmrc");

			if (hasValidNpmToken(repo, npmrc)) {
				step.skip(".npmrc exists with valid token");
				return;
			}

			String token = readAndValidateNpmToken(repo);
			FileUtils.writeShellBlock(npmrc, "%s config".formatted(repo.name), """
					%s:registry=%s/
					//%s/:_authToken=%s
					always-auth=true
					"""
					.formatted(
							repo.name,
							repo.registryUrl,
							repo.registryUrl.getHost() + repo.registryUrl.getPath().replaceAll("/$", ""),
							token
					)
			);
		}
	}

	private static String readAndValidateNpmToken(Repo repo) {
		Optional<String> errorInfo = Optional.empty();

		while (true) {
			Dialog dialog = new Dialog("NPM credentials (%s)".formatted(repo.name));
			Dialog.DialogTextField tokenField = Dialog.passwordField("NPM token");
			dialog.add(tokenField);

			errorInfo
					.map("Error: %s"::formatted)
					.map(Dialog::paragraph)
					.ifPresent(dialog::add);

			String token = dialog.show().get(tokenField);

			try {
				var validationResponse = validateNpmToken(repo, token);
				if (validationResponse.statusCode() == 200) {
					return token;
				}

				errorInfo = Optional.of("%s %s".formatted(validationResponse.statusCode(), validationResponse.body()));
			} catch (IOException e) {
				errorInfo = Optional.of(e.toString());
			}
		}
	}

	private static HttpResponse<String> validateNpmToken(Repo repo, String token) throws IOException {
		try (HttpClient client = HttpClient.newBuilder().connectTimeout(VALIDATION_TIMEOUT).build()) {
			HttpRequest request = HttpRequest.newBuilder()
					.uri(repo.pingUrl)
					.header("Authorization", "Bearer " + token)
					.timeout(VALIDATION_TIMEOUT)
					.GET()
					.build();

			return client.send(request, HttpResponse.BodyHandlers.ofString());
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw new RuntimeException("NPM token validation interrupted", e);
		}
	}

	private static boolean hasValidNpmToken(Repo repo, Path npmrc) {
		Optional<String> existingToken = readExistingNpmToken(repo, npmrc);
		if (existingToken.isEmpty())
			return false;

		try {
			return validateNpmToken(repo, existingToken.get()).statusCode() == 200;
		} catch (IOException ignore) {
			return false;
		}
	}

	private static Optional<String> readExistingNpmToken(Repo repo, Path npmrcPath) {
		try {
			if (!Files.exists(npmrcPath))
				return Optional.empty();

			String content = Files.readString(npmrcPath);


			String quotedHost = Pattern.quote(repo.registryUrl.getHost());
			var matcher = Pattern.compile("//%s.+:_authToken=([^\\s]+)".formatted(quotedHost))
					.matcher(content);

			if (matcher.find()) {
				return Optional.of(matcher.group(1));
			}

			return Optional.empty();
		} catch (IOException e) {
			return Optional.empty();
		}
	}

	public static Repo repo(String name, URI registryUrl, URI pingUrl) {
		return new Repo(name, registryUrl, pingUrl);
	}

	public record Repo(
			String name,
			URI registryUrl,
			URI pingUrl
	) {
	}
}
