/*
 *    Copyright 2019 Ugljesa Jovanovic
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package com.sentray.kmmprotocolmodule.cryptor.common.hash.util

/**
 * Created by Ugljesa Jovanovic
 * ugljesa.jovanovic@ionspin.com
 * on 15-Jul-2019
 */


inline fun <reified T> Array<T>.chunked(sliceSize: Int): Array<Array<T>> {
    val last = this.size % sliceSize
    val hasLast = last != 0
    val numberOfSlices = this.size / sliceSize


    val result: MutableList<List<T>> = MutableList(0) { emptyList() }

    for (i in 0 until numberOfSlices) {
        result.add(this.slice(i * sliceSize until (i + 1) * sliceSize))
    }
    if (hasLast) {
        result.add(this.slice(numberOfSlices * sliceSize until this.size))
    }

    return result.map { it.toTypedArray() }.toTypedArray()

}

internal infix fun UInt.rotateRight(places: Int): UInt {
    return (this shr places) xor (this shl (32 - places))
}

internal infix fun ULong.rotateRight(places: Int): ULong {
    return (this shr places) xor (this shl (64 - places))
}

internal infix fun Array<UByte>.xor(other: Array<UByte>): Array<UByte> {
    if (this.size != other.size) {
        throw RuntimeException("Operands of different sizes are not supported yet")
    }
    return Array(this.size) { this[it] xor other[it] }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal infix fun UByteArray.xor(other: UByteArray): UByteArray {
    if (this.size != other.size) {
        throw RuntimeException("Operands of different sizes are not supported yet")
    }
    return UByteArray(this.size) { this[it] xor other[it] }
}


// UInt / Array utils
internal fun UInt.toBigEndianUByteArray(): Array<UByte> {
    return Array(4) {
        ((this shr (24 - (it * 8))) and 0xFFU).toUByte()
    }
}

internal fun UInt.toLittleEndianTypedUByteArray(): Array<UByte> {
    return Array(4) {
        ((this shr (it * 8)) and 0xFFU).toUByte()
    }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal fun UInt.toLittleEndianUByteArray(): UByteArray {
    return UByteArray(4) {
        ((this shr (it * 8)) and 0xFFU).toUByte()
    }
}

// UInt / Array utils
internal fun ULong.toBigEndianUByteArray(): Array<UByte> {
    return Array(8) {
        ((this shr (56 - (it * 8))) and 0xFFU).toUByte()
    }
}

internal fun ULong.toLittleEndianTypedUByteArray(): Array<UByte> {
    return Array(8) {
        ((this shr (it * 8)) and 0xFFU).toUByte()
    }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal fun ULong.toLittleEndianUByteArray(): UByteArray {
    return UByteArray(8) {
        ((this shr (it * 8)) and 0xFFU).toUByte()
    }
}

internal fun Array<UByte>.fromLittleEndianArrayToULong(): ULong {
    if (this.size > 8) {
        throw RuntimeException("ore than 8 bytes in input, potential overflow")
    }
    return foldIndexed(0UL) { index, acc, uByte -> acc or (uByte.toULong() shl (index * 8)) }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal fun UByteArray.fromLittleEndianArrayToULong(): ULong {
    if (this.size > 8) {
        throw RuntimeException("ore than 8 bytes in input, potential overflow")
    }
    return foldIndexed(0UL) { index, acc, uByte -> acc or (uByte.toULong() shl (index * 8)) }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal fun UByteArray.arrayChunked(sliceSize: Int): List<UByteArray> {
    val last = this.size % sliceSize
    val hasLast = last != 0
    val numberOfSlices = this.size / sliceSize


    val result: MutableList<UByteArray> = MutableList<UByteArray>(0) { ubyteArrayOf() }

    for (i in 0 until numberOfSlices) {
        result.add(this.sliceArray(i * sliceSize until (i + 1) * sliceSize))
    }
    if (hasLast) {
        result.add(this.sliceArray(numberOfSlices * sliceSize until this.size))
    }

    return result
}

internal fun Array<UByte>.fromBigEndianArrayToULong(): ULong {
    if (this.size > 8) {
        throw RuntimeException("ore than 8 bytes in input, potential overflow")
    }
    return foldIndexed(0UL) { index, acc, uByte ->
        val res = acc or (uByte.toULong() shl (56 - (index * 8)))
        res

    }
}

internal fun Array<UByte>.fromLittleEndianArrayToUInt(): UInt {
    if (this.size > 4) {
        throw RuntimeException("ore than 8 bytes in input, potential overflow")
    }
    return foldIndexed(0U) { index, acc, uByte -> acc or (uByte.toUInt() shl (index * 8)) }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal fun UByteArray.fromLittleEndianArrayToUInt(): UInt {
    if (this.size > 4) {
        throw RuntimeException("ore than 8 bytes in input, potential overflow")
    }
    return foldIndexed(0U) { index, acc, uByte -> acc or (uByte.toUInt() shl (index * 8)) }
}

internal fun Array<UByte>.fromBigEndianArrayToUInt(): UInt {
    if (this.size > 4) {
        throw RuntimeException("ore than 8 bytes in input, potential overflow")
    }
    return foldIndexed(0U) { index, acc, uByte -> acc or (uByte.toUInt() shl (24 - (index * 8))) }
}

@OptIn(ExperimentalUnsignedTypes::class)
internal operator fun UInt.plus(other: UByteArray): UByteArray {
    return this.toLittleEndianUByteArray() + other
}

//AES Flatten
@OptIn(ExperimentalUnsignedTypes::class)
internal fun Collection<UByteArray>.flattenToUByteArray(): UByteArray {

    val result = UByteArray(sumOf {
        it.size
    })
    var position = 0
    for (element in this) {
        element.forEach { uByte ->
            result[position] = uByte
            position++
        }
    }
    return result
}
