package com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent

import kotlinx.serialization.Serializable

@Serializable
class ParaStructure(private var raw: ULong = 0UL) {
    enum class RgbwType(val raw: Int) {
        Red(0),
        Green(1),
        Blue(2),
        Bright(3),
        W(4)
    }

    enum class IrValueType {
        Cmd,
        Index
    }

    enum class IrCmd(val raw: Int) {
        Send(1),
        Learn(2),
        Delete(3)
    }

    enum class CurtainAction(val raw: Int) {
        Stop(0),
        Close(1),
        Open(2)
    }

    enum class CurtainWithLocAction(val raw: Int) {
        Close(0),
        Open(100),
        ChangeDirection(254),
        Stop(255),

        @Deprecated("电机无法通过命令初始化，需按照包装盒内的说明书操作")
        Initial(253),
    }

    enum class RgbwAction(val raw: Int) {
        RgbwModeRandomQuick(251),
        RgbwModeSwitchSlow(250),
        RgbwModeSwitchQuick(249),
        RgbwModeRandomSlow(248);

        companion object {
            private val mapping = values().associateBy(RgbwAction::raw)
            fun fromRaw(raw: Int) = mapping[raw]
        }
    }

    enum class ParaAction(val raw: Int) {
        OnOff(254)
    }

    fun getRawValue(): ULong {
        return this.raw
    }

    fun getColor(rgbw: RgbwType): Int {
        return getValue(rgbw.raw).toInt()
    }

    fun setColor(rgbw: RgbwType, value: Int) {
        setValue(rgbw.raw, value.toULong())
    }

    fun getRGBWAction(): RgbwAction? {
        return RgbwAction.fromRaw(getColor(RgbwType.Bright).toInt())
    }

    fun setRGBWAction(rgbwAction: RgbwAction) {
        setColor(RgbwType.Red, 0)
        setColor(RgbwType.Green, 0)
        setColor(RgbwType.Blue, 0)
        setColor(RgbwType.W, 0)
        setColor(RgbwType.Bright, rgbwAction.raw)
    }

    fun getIRPara(ir: IrValueType): ULong {
        return when (ir) {
            IrValueType.Cmd -> this.raw shr 12
            IrValueType.Index -> (this.raw and 0x0f.toULong())
        }
    }

    fun setIRPara(ir: IrValueType, value: ULong) {
        when (ir) {
            IrValueType.Cmd -> {
                this.raw = this.raw and 0x0f.toULong()
                this.raw = this.raw or (value shl 12)
            }
            IrValueType.Index -> {
                this.raw = this.raw and 0x0f.toULong().inv()
                this.raw += value
            }
        }
    }

    fun getTemperatureLampBright(): Long {
        return getValue(0).toLong()
    }

    fun getTemperatureLampTemp(): Long {
        return getValue(1).toLong()
    }

    fun setTemperatureLampBright(value: ULong) {
        setValue(0, value and 0xffUL)
    }

    fun setTemperatureLampTemp(value: ULong) {
        setValue(1, value and 0xffUL)
    }

//    fun freezeSelf() {
//        freeze(this)
//    }

    private fun setValue(paraIndex: Int, valueNew: ULong) {
        val leftShiftCount = paraIndex * 8
        val byteMask: ULong = (0xffUL shl leftShiftCount).inv() //0xffffff00ffffff

        this.raw = this.raw and byteMask    //将待赋值的 bit 清零
        val valueNewShift = valueNew shl leftShiftCount
        this.raw = this.raw or valueNewShift    //赋值
    }

    private fun getValue(paraIndex: Int): ULong {
        val rightShiftCount = (paraIndex * 8)
        return ((raw shr rightShiftCount) and 0xff.toULong())
    }
}