/*
 * Copyright 2013 Luca Longinotti <l@longi.li>
 * See LICENSE.md for licensing information.
 *
 * Based on libusb <http://libusb.info/>:
 *
 * Copyright 2001 Johannes Erdfelt <johannes@erdfelt.com>
 * Copyright 2007-2009 Daniel Drake <dsd@gentoo.org>
 * Copyright 2010-2012 Peter Stuge <peter@stuge.se>
 * Copyright 2008-2013 Nathan Hjelm <hjelmn@users.sourceforge.net>
 * Copyright 2009-2013 Pete Batard <pete@akeo.ie>
 * Copyright 2009-2013 Ludovic Rousseau <ludovic.rousseau@gmail.com>
 * Copyright 2010-2012 Michael Plante <michael.plante@gmail.com>
 * Copyright 2011-2013 Hans de Goede <hdegoede@redhat.com>
 * Copyright 2012-2013 Martin Pieuchot <mpi@openbsd.org>
 * Copyright 2012-2013 Toby Gray <toby.gray@realvnc.com>
 */

package org.usb4java;

/**
 * Hotplug Callback Handle.
 *
 * Callback handles are generated by {@link LibUsb#hotplugRegisterCallback(
 * Context, int, int, int, int, int, HotplugCallback, Object, 
 * HotplugCallbackHandle)} and can be used to deregister callbacks. Callback 
 * handles are unique per {@link Context} and it is safe to call 
 * {@link LibUsb#hotplugDeregisterCallback(Context, HotplugCallbackHandle)}
 * on an already deregistered callback.
 *
 * @author Luca Longinotti (l@longi.li)
 */
public final class HotplugCallbackHandle
{
    /** The hotplug callback handle, it's an integer (int) in C. */
    private long hotplugCallbackHandleValue;

    /**
     * Constructs a new hotplug callback handle. Must be passed to
     * {@link LibUsb#hotplugRegisterCallback(Context, int, int, int, int, int, 
     * HotplugCallback, Object, HotplugCallbackHandle)} before passing it
     * to any other method.
     */
    public HotplugCallbackHandle()
    {
        // Empty
    }

    /**
     * Returns the hotplug callback handle value.
     *
     * @return The hotplug callback handle value.
     */
    public long getValue()
    {
        return this.hotplugCallbackHandleValue;
    }

    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = 1;
        result = (prime * result) + (int) (this.hotplugCallbackHandleValue ^ 
            (this.hotplugCallbackHandleValue >>> 32));
        return result;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (this.getClass() != obj.getClass())
        {
            return false;
        }
        final HotplugCallbackHandle other = (HotplugCallbackHandle) obj;
        if (this.hotplugCallbackHandleValue != other.hotplugCallbackHandleValue)
        {
            return false;
        }
        return true;
    }

    @Override
    public String toString()
    {
        return String.format("libusb hotplug callback handle 0x%x",
            this.hotplugCallbackHandleValue);
    }
}
