package com.github.elebras1.flecs.processor;


import com.palantir.javapoet.*;

import javax.lang.model.element.*;
import java.lang.foreign.Arena;
import java.lang.foreign.MemoryLayout;
import java.lang.foreign.MemorySegment;
import java.util.List;

public class ComponentCodeGenerator {

    private static final String LAYOUT_FIELD_CLASS = "com.github.elebras1.flecs.util.LayoutField";
    private static final String COMPONENT_INTERFACE = "com.github.elebras1.flecs.Component";

    public JavaFile generateComponentClass(TypeElement recordElement, List<VariableElement> fields) {
        String packageName = this.getPackageName(recordElement);
        String recordName = recordElement.getSimpleName().toString();
        String componentClassName = recordName + "Component";

        TypeSpec componentClass = TypeSpec.classBuilder(componentClassName)
                .addModifiers(Modifier.PUBLIC, Modifier.FINAL)
                .addSuperinterface(ParameterizedTypeName.get(ClassName.bestGuess(COMPONENT_INTERFACE), ClassName.get(packageName, recordName)))
                .addField(this.createLayoutField(recordName, fields))
                .addFields(this.createOffsetFields(fields))
                .addType(this.createSingletonHolder(packageName, recordName))
                .addMethod(this.createConstructor())
                .addMethod(this.createLayoutMethod())
                .addMethod(this.createWriteMethod(recordName, fields))
                .addMethod(this.createReadMethod(packageName, recordName, fields))
                .addMethod(this.createArrayMethod(packageName, recordName))
                .addMethod(this.createOffsetOfMethod(fields))
                .addMethod(this.createFactoryMethod(packageName, recordName))
                .addMethod(this.createGetInstanceMethod(packageName, recordName))
                .build();

        return JavaFile.builder(packageName, componentClass)
                .addFileComment("Generated by FlecsComponentProcessor")
                .indent("    ")
                .build();
    }

    private String getPackageName(TypeElement element) {
        Element current = element.getEnclosingElement();
        while (current != null && !(current instanceof PackageElement)) {
            current = current.getEnclosingElement();
        }
        return current != null ? ((PackageElement) current).getQualifiedName().toString() : "";
    }

    private FieldSpec createLayoutField(String recordName, List<VariableElement> fields) {
        CodeBlock.Builder layoutBuilder = CodeBlock.builder()
                .add("$T.structLayout(\n", MemoryLayout.class)
                .indent();

        for (int i = 0; i < fields.size(); i++) {
            VariableElement field = fields.get(i);
            String fieldName = field.getSimpleName().toString();
            String layoutMethod = getLayoutMethod(field.asType().toString());

            layoutBuilder.add("$L.$L().withName($S)", LAYOUT_FIELD_CLASS, layoutMethod, fieldName);
            if (i < fields.size() - 1) {
                layoutBuilder.add(",\n");
            }
        }

        layoutBuilder.unindent().add("\n).withName($S)", recordName);

        return FieldSpec.builder(MemoryLayout.class, "LAYOUT", Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                .initializer(layoutBuilder.build())
                .build();
    }

    private List<FieldSpec> createOffsetFields(List<VariableElement> fields) {
        return fields.stream()
                .map(field -> {
                    String fieldName = field.getSimpleName().toString();
                    String constantName = "OFFSET_" + fieldName.toUpperCase();
                    return FieldSpec.builder(long.class, constantName, Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                            .initializer("$L.offsetOf(LAYOUT, $S)", LAYOUT_FIELD_CLASS, fieldName)
                            .build();
                })
                .toList();
    }

    private MethodSpec createConstructor() {
        return MethodSpec.constructorBuilder()
                .addModifiers(Modifier.PUBLIC)
                .build();
    }

    private MethodSpec createLayoutMethod() {
        return MethodSpec.methodBuilder("layout")
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .returns(MemoryLayout.class)
                .addStatement("return LAYOUT")
                .build();
    }

    private MethodSpec createWriteMethod(String recordName, List<VariableElement> fields) {
        MethodSpec.Builder method = MethodSpec.methodBuilder("write")
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .addParameter(MemorySegment.class, "segment")
                .addParameter(TypeVariableName.get(recordName), "data")
                .addParameter(Arena.class, "arena");

        for (VariableElement field : fields) {
            String fieldName = field.getSimpleName().toString();
            String offsetName = "OFFSET_" + fieldName.toUpperCase();
            String typeName = field.asType().toString();
            if ("java.lang.String".equals(typeName)) {
                method.addStatement("$L.set(segment, $L, data.$L(), arena)", LAYOUT_FIELD_CLASS, offsetName, fieldName);
            } else {
                method.addStatement("$L.set(segment, $L, data.$L())", LAYOUT_FIELD_CLASS, offsetName, fieldName);
            }
        }

        return method.build();
    }

    private MethodSpec createReadMethod(String packageName, String recordName, List<VariableElement> fields) {
        MethodSpec.Builder method = MethodSpec.methodBuilder("read")
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .returns(ClassName.get(packageName, recordName))
                .addParameter(MemorySegment.class, "segment");

        for (VariableElement field : fields) {
            String fieldName = field.getSimpleName().toString();
            String offsetName = "OFFSET_" + fieldName.toUpperCase();
            String typeName = field.asType().toString();
            String getterMethod = getGetterMethod(typeName);

            method.addStatement("$L $L = $L.$L(segment, $L)", typeName, fieldName, LAYOUT_FIELD_CLASS, getterMethod, offsetName);
        }

        CodeBlock.Builder returnStatement = CodeBlock.builder().add("return new $L(", recordName);

        for (int i = 0; i < fields.size(); i++) {
            returnStatement.add(fields.get(i).getSimpleName().toString());
            if (i < fields.size() - 1) {
                returnStatement.add(", ");
            }
        }
        returnStatement.add(")");

        method.addStatement(returnStatement.build());
        return method.build();
    }

    private MethodSpec createFactoryMethod(String packageName, String recordName) {
        MethodSpec.Builder method = MethodSpec.methodBuilder("create")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(ClassName.get(packageName, recordName + "Component"));

        method.addStatement("return new $LComponent()", recordName);
        return method.build();
    }

    private TypeSpec createSingletonHolder(String packageName, String recordName) {
        String componentClassName = recordName + "Component";
        return TypeSpec.classBuilder("Holder")
                .addModifiers(Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                .addField(FieldSpec.builder(ClassName.get(packageName, componentClassName), "INSTANCE", Modifier.STATIC, Modifier.FINAL)
                        .initializer("new $L()", componentClassName)
                        .build())
                .build();
    }

    private MethodSpec createGetInstanceMethod(String packageName, String recordName) {
        String componentClassName = recordName + "Component";
        return MethodSpec.methodBuilder("getInstance")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(ClassName.get(packageName, componentClassName))
                .addStatement("return Holder.INSTANCE")
                .build();
    }

    private MethodSpec createArrayMethod(String packageName, String recordName) {
        TypeName recordType = ClassName.get(packageName, recordName);
        TypeName arrayType = ArrayTypeName.of(recordType);

        return MethodSpec.methodBuilder("createArray")
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .addParameter(int.class, "size")
                .returns(arrayType)
                .addStatement("return new $T[size]", recordType)
                .build();
    }

    private String getLayoutMethod(String type) {
        return switch (type) {
            case "short" -> "shortLayout";
            case "int" -> "intLayout";
            case "long" -> "longLayout";
            case "float" -> "floatLayout";
            case "double" -> "doubleLayout";
            case "boolean" -> "booleanLayout";
            case "java.lang.String" -> "stringLayout";
            default -> throw new IllegalArgumentException("Unsupported type: " + type);
        };
    }

    private String getGetterMethod(String type) {
        return switch (type) {
            case "short" -> "getShort";
            case "int" -> "getInt";
            case "long" -> "getLong";
            case "float" -> "getFloat";
            case "double" -> "getDouble";
            case "boolean" -> "getBoolean";
            case "java.lang.String" -> "getString";
            default -> throw new IllegalArgumentException("Unsupported type: " + type);
        };
    }

    private MethodSpec createOffsetOfMethod(List<VariableElement> fields) {
        MethodSpec.Builder method = MethodSpec.methodBuilder("offsetOf")
                .addAnnotation(Override.class)
                .addModifiers(Modifier.PUBLIC)
                .addParameter(String.class, "fieldName")
                .returns(long.class);

        if (fields.isEmpty()) {
            method.addStatement("throw new $T($S + fieldName)", IllegalArgumentException.class, "Unknown field: ");
        } else {
            method.beginControlFlow("return switch (fieldName)");
            for (VariableElement field : fields) {
                String fieldName = field.getSimpleName().toString();
                String offsetName = "OFFSET_" + fieldName.toUpperCase();
                method.addStatement("case $S -> $L", fieldName, offsetName);
            }
            method.addStatement("default -> throw new $T($S + fieldName)", IllegalArgumentException.class, "Unknown field: ");
            method.endControlFlow("");
        }

        return method.build();
    }
}

