package com.github.elebras1.flecs.processor;

import com.palantir.javapoet.*;

import javax.lang.model.element.Modifier;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.Element;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ComponentMapGenerator {

    private static final ClassName COMPONENT_INTERFACE = ClassName.get("com.github.elebras1.flecs", "Component");
    private static final String MAP_PACKAGE = "com.github.elebras1.flecs";
    private static final String MAP_CLASS_NAME = "ComponentMap";

    public JavaFile generateComponentMap(List<TypeElement> components) {
        TypeSpec.Builder mapClass = TypeSpec.classBuilder(MAP_CLASS_NAME)
                .addModifiers(Modifier.PUBLIC, Modifier.FINAL)
                .addField(this.createMapField())
                .addStaticBlock(this.createStaticInitializer(components))
                .addMethod(this.createConstructor())
                .addMethod(this.createGetInstanceMethod());

        return JavaFile.builder(MAP_PACKAGE, mapClass.build())
                .addFileComment("Generated by ComponentMapGenerator.")
                .indent("    ")
                .build();
    }

    private FieldSpec createMapField() {
        ParameterizedTypeName mapType = ParameterizedTypeName.get(
                ClassName.get(Map.class),
                ParameterizedTypeName.get(ClassName.get(Class.class), WildcardTypeName.subtypeOf(Object.class)),
                ParameterizedTypeName.get(COMPONENT_INTERFACE, WildcardTypeName.subtypeOf(Object.class))
        );

        return FieldSpec.builder(mapType, "COMPONENT_MAP", Modifier.PRIVATE, Modifier.STATIC, Modifier.FINAL)
                .initializer("new $T<>()", HashMap.class)
                .build();
    }

    private CodeBlock createStaticInitializer(List<TypeElement> components) {
        CodeBlock.Builder builder = CodeBlock.builder();

        for (TypeElement component : components) {
            String packageName = this.getPackageName(component);
            String recordName = component.getSimpleName().toString();
            String componentClassName = recordName + "Component";

            ClassName recordClass = ClassName.get(packageName, recordName);
            ClassName componentClass = ClassName.get(packageName, componentClassName);

            builder.addStatement("COMPONENT_MAP.put($T.class, $T.getInstance())", recordClass, componentClass);
        }

        return builder.build();
    }

    private MethodSpec createConstructor() {
        return MethodSpec.constructorBuilder()
                .addModifiers(Modifier.PRIVATE)
                .addComment("Private constructor to prevent instantiation")
                .build();
    }

    private MethodSpec createGetInstanceMethod() {
        MethodSpec.Builder method = MethodSpec.methodBuilder("getInstance")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .addTypeVariable(TypeVariableName.get("T"))
                .addParameter(ParameterizedTypeName.get(ClassName.get(Class.class), TypeVariableName.get("T")), "componentClass")
                .returns(ParameterizedTypeName.get(COMPONENT_INTERFACE, TypeVariableName.get("T")))
                .addAnnotation(AnnotationSpec.builder(SuppressWarnings.class)
                        .addMember("value", "$S", "unchecked")
                        .build());

        method.addStatement("return ($T<T>) COMPONENT_MAP.get(componentClass)", COMPONENT_INTERFACE);

        return method.build();
    }

    private String getPackageName(TypeElement element) {
        Element current = element.getEnclosingElement();
        while (current != null && !(current instanceof PackageElement)) {
            current = current.getEnclosingElement();
        }
        return current != null ? ((PackageElement) current).getQualifiedName().toString() : "";
    }
}

