/*
 * Decompiled with CFR 0.152.
 */
package io.github.hyperliquid.sdk;

import io.github.hyperliquid.sdk.apis.Exchange;
import io.github.hyperliquid.sdk.apis.Info;
import io.github.hyperliquid.sdk.model.wallet.ApiWallet;
import io.github.hyperliquid.sdk.utils.HypeError;
import io.github.hyperliquid.sdk.utils.HypeHttpClient;
import java.math.BigInteger;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.ECKeyPair;
import org.web3j.utils.Numeric;

public class HyperliquidClient {
    private static final Logger log = LoggerFactory.getLogger(HyperliquidClient.class);
    private final Info info;
    private final Map<String, Exchange> exchangesByAddress;
    private final List<ApiWallet> apiWallets;

    public HyperliquidClient(Info info, Map<String, Exchange> exchangesByAddress, List<ApiWallet> apiWallets) {
        this.info = info;
        this.exchangesByAddress = exchangesByAddress;
        this.apiWallets = apiWallets;
    }

    public Info getInfo() {
        return this.info;
    }

    public Map<String, Exchange> getExchangesByAddress() {
        return this.exchangesByAddress;
    }

    public List<ApiWallet> getApiWallets() {
        return this.apiWallets;
    }

    @Deprecated
    public Exchange getSingleExchange() {
        return this.getExchange();
    }

    public Exchange getExchange() {
        if (this.exchangesByAddress.isEmpty()) {
            throw new HypeError("No exchange instances available.");
        }
        return this.exchangesByAddress.values().iterator().next();
    }

    @Deprecated
    public Exchange useExchange(String address) {
        return this.getExchange(address);
    }

    public Exchange getExchange(String address) {
        if (address == null || address.trim().isEmpty()) {
            throw new HypeError("Wallet address cannot be null or empty.");
        }
        Exchange ex = this.exchangesByAddress.get(address);
        if (ex == null) {
            String availableAddresses = String.join((CharSequence)", ", this.exchangesByAddress.keySet());
            throw new HypeError(String.format("Wallet address '%s' not found. Available addresses: [%s]", address, availableAddresses.isEmpty() ? "none" : availableAddresses));
        }
        return ex;
    }

    public boolean hasWallet(String address) {
        return address != null && this.exchangesByAddress.containsKey(address);
    }

    public Set<String> getAvailableAddresses() {
        return Collections.unmodifiableSet(this.exchangesByAddress.keySet());
    }

    public List<String> listWallets() {
        return new ArrayList<String>(this.exchangesByAddress.keySet());
    }

    public Exchange getExchangeByIndex(int index) {
        List<String> addresses = this.listWallets();
        if (index < 0 || index >= addresses.size()) {
            throw new HypeError(String.format("Wallet index %d out of bounds. Valid range: [0, %d]", index, addresses.size() - 1));
        }
        return this.exchangesByAddress.get(addresses.get(index));
    }

    public int getWalletCount() {
        return this.exchangesByAddress.size();
    }

    public String getSingleAddress() {
        if (this.apiWallets == null || this.apiWallets.isEmpty()) {
            throw new HypeError("No wallets available. Please add at least one wallet.");
        }
        return this.apiWallets.getFirst().getPrimaryWalletAddress();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String baseUrl = "https://api.hyperliquid.xyz";
        private int timeout = 10;
        private boolean skipWs = false;
        private final List<ApiWallet> apiWallets = new ArrayList<ApiWallet>();
        private OkHttpClient okHttpClient = null;
        private boolean autoWarmUpCache = true;

        public Builder baseUrl(String baseUrl) {
            this.baseUrl = baseUrl;
            return this;
        }

        public Builder testNetUrl() {
            this.baseUrl = "https://api.hyperliquid-testnet.xyz";
            return this;
        }

        public Builder addPrivateKey(String privateKey) {
            this.addApiWallet(null, privateKey);
            return this;
        }

        public Builder addPrivateKeys(List<String> pks) {
            for (String pk : pks) {
                this.addPrivateKey(pk);
            }
            return this;
        }

        public Builder skipWs(boolean skipWs) {
            this.skipWs = skipWs;
            return this;
        }

        public Builder timeout(int timeout) {
            this.timeout = timeout;
            return this;
        }

        public Builder addApiWallet(ApiWallet apiWallet) {
            this.apiWallets.add(apiWallet);
            return this;
        }

        public Builder addApiWallet(String primaryWalletAddress, String apiWalletPrivateKey) {
            this.apiWallets.add(new ApiWallet(primaryWalletAddress, apiWalletPrivateKey));
            return this;
        }

        public Builder addApiWallets(List<ApiWallet> apiWallets) {
            this.apiWallets.addAll(apiWallets);
            return this;
        }

        public Builder okHttpClient(OkHttpClient client) {
            this.okHttpClient = client;
            return this;
        }

        public Builder disableAutoWarmUpCache() {
            this.autoWarmUpCache = false;
            return this;
        }

        private OkHttpClient getOkHttpClient() {
            return this.okHttpClient != null ? this.okHttpClient : new OkHttpClient.Builder().connectTimeout(Duration.ofSeconds(this.timeout)).readTimeout(Duration.ofSeconds(this.timeout)).writeTimeout(Duration.ofSeconds(this.timeout)).build();
        }

        public HyperliquidClient build() {
            OkHttpClient httpClient = this.getOkHttpClient();
            HypeHttpClient hypeHttpClient = new HypeHttpClient(this.baseUrl, httpClient);
            Info info = new Info(this.baseUrl, hypeHttpClient, this.skipWs);
            LinkedHashMap<String, Exchange> exchangesByAddress = new LinkedHashMap<String, Exchange>();
            if (!this.apiWallets.isEmpty()) {
                for (ApiWallet apiWallet : this.apiWallets) {
                    this.validatePrivateKey(apiWallet.getApiWalletPrivateKey());
                    Credentials credentials = Credentials.create((String)apiWallet.getApiWalletPrivateKey());
                    apiWallet.setCredentials(credentials);
                    if (apiWallet.getPrimaryWalletAddress() == null || apiWallet.getPrimaryWalletAddress().trim().isEmpty()) {
                        apiWallet.setPrimaryWalletAddress(credentials.getAddress());
                    }
                    exchangesByAddress.put(apiWallet.getPrimaryWalletAddress(), new Exchange(hypeHttpClient, apiWallet, info));
                }
            }
            if (this.autoWarmUpCache) {
                try {
                    info.warmUpCache();
                }
                catch (Exception e) {
                    log.warn("[HyperliquidClient] Warning: Cache warm-up failed, but client is still usable. First API calls may be slower. Error: {}", (Object)e.getMessage());
                }
            }
            return new HyperliquidClient(info, Collections.unmodifiableMap(exchangesByAddress), Collections.unmodifiableList(new ArrayList<ApiWallet>(this.apiWallets)));
        }

        private void validatePrivateKey(String privateKey) {
            String normalizedKey;
            if (privateKey == null || privateKey.trim().isEmpty()) {
                throw new HypeError("Private key cannot be null or empty.");
            }
            String string = normalizedKey = privateKey.startsWith("0x") ? privateKey.substring(2) : privateKey;
            if (!normalizedKey.matches("^[0-9a-fA-F]+$")) {
                throw new HypeError("Private key contains invalid characters. Must be hex.");
            }
            if (normalizedKey.length() != 64) {
                throw new HypeError("Private key must be 64 hexadecimal characters long.");
            }
            try {
                BigInteger keyInt = Numeric.toBigInt((String)privateKey);
                ECKeyPair.create((BigInteger)keyInt);
            }
            catch (Exception e) {
                throw new HypeError("Invalid private key: cryptographic validation failed.", e);
            }
        }
    }
}

