package io.github.hyperliquid.sdk.model.order;

import java.util.ArrayList;
import java.util.List;

/**
 * Open position + take-profit/stop-loss combined order builder.
 * <p>
 * Used to build normalTpsl or positionTpsl order groups, simplifying one-click setup of take-profit/stop-loss operations.
 * <p>
 * Usage examples:
 * <pre>
 * // 1. Limit order open + take-profit/stop-loss (normalTpsl)
 * OrderGroup orderGroup = OrderRequest.entryWithTpSl()
 *     .perp("ETH")
 *     .buy(0.1)
 *     .entryPrice(3500.0)  // Set entryPrice, will use limit order
 *     .takeProfit(3600.0)
 *     .stopLoss(3400.0)
 *     .buildNormalTpsl();
 *
 * // Automatically infer grouping="normalTpsl" when submitting
 * JsonNode result = exchange.bulkOrders(orderGroup);
 *
 * // 2. Market order open + take-profit/stop-loss (normalTpsl)
 * OrderGroup orderGroup = OrderRequest.entryWithTpSl()
 *     .perp("ETH")
 *     .buy(0.1)
 *     // No entryPrice set, will use market order
 *     .takeProfit(3600.0)
 *     .stopLoss(3400.0)
 *     .buildNormalTpsl();
 *
 * // Automatically infer grouping="normalTpsl" when submitting
 * JsonNode result = exchange.bulkOrders(orderGroup);
 *
 * // 3. Add take-profit/stop-loss to existing position (positionTpsl - manual specification)
 * OrderGroup orderGroup = OrderRequest.entryWithTpSl()
 *     .perp("ETH")
 *     .closePosition(0.5, true)  // Close 0.5 ETH long position
 *     .takeProfit(3600.0)
 *     .stopLoss(3400.0)
 *     .buildPositionTpsl();
 *
 * // Automatically infer grouping="positionTpsl" when submitting
 * JsonNode result = exchange.bulkOrders(orderGroup);
 *
 * // 4. Add take-profit/stop-loss to existing position (positionTpsl - automatic inference)
 * OrderGroup orderGroup = OrderRequest.entryWithTpSl()
 *     .perp("ETH")
 *     // Don't call closePosition(), Exchange will automatically query account positions and infer direction and quantity
 *     .takeProfit(3600.0)
 *     .stopLoss(3400.0)
 *     .buildPositionTpsl();
 *
 * // Exchange will automatically call API to get position information and fill direction and quantity
 * JsonNode result = exchange.bulkOrders(orderGroup);
 * </pre>
 */
public class OrderWithTpSlBuilder {
    /**
     * Instrument type (PERP or SPOT)
     */
    private InstrumentType instrumentType;

    /**
     * Currency name
     */
    private String coin;

    /**
     * Whether to buy (true=buy/long, false=sell/short)
     */
    private Boolean isBuy;

    /**
     * Order quantity (string)
     */
    private String sz;

    /**
     * Entry limit price (string, required for buildAll mode)
     */
    private String entryPrice;

    /**
     * Take-profit price (string)
     */
    private String takeProfitPrice;

    /**
     * Stop-loss price (string)
     */
    private String stopLossPrice;

    /**
     * Entry order TIF strategy (default GTC)
     */
    private Tif entryTif = Tif.GTC;

    /**
     * Client order ID
     */
    private Cloid cloid;

    /**
     * Order expiration time (milliseconds)
     */
    private Long expiresAfter;

    public OrderWithTpSlBuilder() {
    }

    // ========================================
    // 5. 过期时间
    // ========================================

    /**
     * 设置订单过期时间。
     *
     * @param expiresAfter 过期时间（毫秒）
     * @return this
     */
    public OrderWithTpSlBuilder expiresAfter(Long expiresAfter) {
        this.expiresAfter = expiresAfter;
        return this;
    }

    // ========================================
    // 1. 交易品种
    // ========================================

    /**
     * 设置为永续合约。
     *
     * @param coin 币种名称
     * @return this
     */
    public OrderWithTpSlBuilder perp(String coin) {
        this.instrumentType = InstrumentType.PERP;
        this.coin = coin;
        return this;
    }

    /**
     * 设置为现货。
     *
     * @param coin 币种名称
     * @return this
     */
    public OrderWithTpSlBuilder spot(String coin) {
        this.instrumentType = InstrumentType.SPOT;
        this.coin = coin;
        return this;
    }

    // ========================================
    // 2. 方向与数量
    // ========================================

    /**
     * 买入开仓。
     *
     * @param sz 数量（字符串）
     * @return this
     */
    public OrderWithTpSlBuilder buy(String sz) {
        this.isBuy = true;
        this.sz = sz;
        return this;
    }

    /**
     * 卖出开仓。
     *
     * @param sz 数量（字符串）
     * @return this
     */
    public OrderWithTpSlBuilder sell(String sz) {
        this.isBuy = false;
        this.sz = sz;
        return this;
    }

    /**
     * 平仓模式（用于 positionTpsl）。
     * <p>
     * 当你已经持有仓位，想为仓位添加止盈止损时使用。
     *
     * @param sz             仓位数量（字符串）
     * @param isLongPosition 是否多仓（true=多仓，false=空仓）
     * @return this
     */
    public OrderWithTpSlBuilder closePosition(String sz, boolean isLongPosition) {
        this.isBuy = isLongPosition; // 多仓需要卖出平仓，空仓需要买入平仓
        this.sz = sz;
        this.entryPrice = null; // 不需要开仓价
        return this;
    }

    // ========================================
    // 3. 价格设置
    // ========================================

    /**
     * 设置开仓限价。
     *
     * @param entryPrice 开仓价格（字符串）
     * @return this
     */
    public OrderWithTpSlBuilder entryPrice(String entryPrice) {
        this.entryPrice = entryPrice;
        return this;
    }

    /**
     * 设置止盈价格。
     *
     * @param tpPrice 止盈价（字符串）
     * @return this
     */
    public OrderWithTpSlBuilder takeProfit(String tpPrice) {
        this.takeProfitPrice = tpPrice;
        return this;
    }

    /**
     * 设置止损价格。
     *
     * @param slPrice 止损价（字符串）
     * @return this
     */
    public OrderWithTpSlBuilder stopLoss(String slPrice) {
        this.stopLossPrice = slPrice;
        return this;
    }

    // ========================================
    // 4. 其他选项
    // ========================================

    /**
     * 设置开仓单的 TIF 策略。
     *
     * @param tif TIF 策略
     * @return this
     */
    public OrderWithTpSlBuilder entryTif(Tif tif) {
        this.entryTif = tif;
        return this;
    }

    /**
     * 设置客户端订单 ID。
     *
     * @param cloid Cloid
     * @return this
     */
    public OrderWithTpSlBuilder cloid(Cloid cloid) {
        this.cloid = cloid;
        return this;
    }

    // ========================================
    // 5. 构建
    // ========================================

    /**
     * 构建 normalTpsl 订单组（开仓 + TP + SL）。
     * <p>
     * 返回的 OrderGroup 会自动携带 grouping="normalTpsl" 类型信息。
     * 在调用 exchange.bulkOrders(orderGroup) 时会自动推断使用 normalTpsl 分组。
     *
     * @return OrderGroup 包含订单列表和分组类型
     * @throws IllegalStateException 当必填字段缺失时抛出
     */
    public OrderGroup buildNormalTpsl() {
        return new OrderGroup(buildOrderList(true), GroupingType.NORMAL_TPSL);
    }

    /**
     * 构建 positionTpsl 订单组（仅 TP + SL，不含开仓单）。
     * <p>
     * 返回的 OrderGroup 会自动携带 grouping="positionTpsl" 类型信息。
     * 用于为已有仓位添加或修改止盈止损。
     * <p>
     * <b>自动推断仓位功能：</b>
     * <ul>
     *   <li>如果调用了 closePosition(sz, isLong) - 手动指定仓位方向和数量</li>
     *   <li>如果未调用 closePosition() - Exchange 会自动查询账户持仓并推断方向和数量</li>
     * </ul>
     *
     * @return OrderGroup 包含订单列表和分组类型
     * @throws IllegalStateException 当必填字段缺失时抛出
     */
    public OrderGroup buildPositionTpsl() {
        return new OrderGroup(buildOrderList(false), GroupingType.POSITION_TPSL);
    }

    /**
     * 构建订单列表。
     *
     * @param includeEntry 是否包含开仓单（true=normalTpsl，false=positionTpsl）
     * @return 订单列表
     */
    private List<OrderRequest> buildOrderList(boolean includeEntry) {
        validate(includeEntry);
        List<OrderRequest> orders = new ArrayList<>();
        
        // 1. 开仓单（仅在 normalTpsl 模式下添加）
        if (includeEntry) {
            OrderRequest entry;
            if (entryPrice == null) {
            // 市价单 - 创建一个占位请求，后续在Exchange中会通过marketOpenTransition方法处理
            entry = new OrderRequest();
            entry.setInstrumentType(instrumentType != null ? instrumentType : InstrumentType.PERP);
            entry.setCoin(coin);
            entry.setIsBuy(isBuy);
            entry.setSz(sz);
            entry.setReduceOnly(false);
            entry.setCloid(cloid);
            // 设置为 IOC 市价单，价格将在Exchange.marketOpenTransition中计算
            LimitOrderType limitOrderType = new LimitOrderType(Tif.IOC);
            OrderType orderType = new OrderType(limitOrderType);
            entry.setOrderType(orderType);
        } else {
            // 限价单
            entry = new OrderRequest(
                    instrumentType != null ? instrumentType : InstrumentType.PERP,
                    coin,
                    isBuy,
                    sz,
                    entryPrice,
                    new OrderType(new LimitOrderType(entryTif)),
                    false,
                    cloid
            );
        }
            if (expiresAfter != null) {
                entry.setExpiresAfter(expiresAfter);
            }
            orders.add(entry);
        }
        // 2. 止盈单（平仓）
        if (takeProfitPrice != null) {
            OrderRequest tp = new OrderRequest(
                    instrumentType != null ? instrumentType : InstrumentType.PERP,
                    coin,
                    isBuy != null ? !isBuy : null,  // 反向平仓（如果 isBuy 为 null，则保持 null）
                    sz,
                    takeProfitPrice,
                    new OrderType(new TriggerOrderType(takeProfitPrice, true, TriggerOrderType.TpslType.TP)),
                    true,
                    null
            );

            if (expiresAfter != null) {
                tp.setExpiresAfter(expiresAfter);
            }
            orders.add(tp);
        }

        // 3. 止损单（平仓）
        if (stopLossPrice != null) {
            OrderRequest sl = new OrderRequest(
                    instrumentType != null ? instrumentType : InstrumentType.PERP,
                    coin,
                    isBuy != null ? !isBuy : null,
                    sz,
                    stopLossPrice,
                    new OrderType(new TriggerOrderType(stopLossPrice, true, TriggerOrderType.TpslType.SL)),
                    true,
                    null
            );

            if (expiresAfter != null) {
                sl.setExpiresAfter(expiresAfter);
            }
            orders.add(sl);
        }

        return orders;
    }

    /**
     * 校验必填字段。
     *
     * @param isNormalTpsl 是否为 normalTpsl 模式（true=normalTpsl，false=positionTpsl）
     */
    private void validate(boolean isNormalTpsl) {
        if (coin == null || coin.isEmpty()) {
            throw new IllegalStateException("coin is required");
        }
        
        // normalTpsl 模式：必须指定方向和数量
        if (isNormalTpsl) {
            if (isBuy == null) {
                throw new IllegalStateException("direction is required for normalTpsl (call buy() or sell())");
            }
            if (sz == null || sz.isEmpty()) {
                throw new IllegalStateException("size is required for normalTpsl (call buy() or sell())");
            }
        }
        // positionTpsl 模式：允许 isBuy 和 sz 为 null（由 Exchange 自动推断）
        // 但如果设置了 sz，则不能为空字符串
        else {
            if (sz != null && sz.isEmpty()) {
                throw new IllegalStateException("size cannot be empty if specified");
            }
        }
        
        if (takeProfitPrice == null && stopLossPrice == null) {
            throw new IllegalStateException("at least one of takeProfit or stopLoss is required");
        }
    }
}
