/*
 * Decompiled with CFR 0.152.
 */
package io.github.hyperliquid.sdk.websocket;

import com.fasterxml.jackson.databind.JsonNode;
import io.github.hyperliquid.sdk.model.subscription.Subscription;
import io.github.hyperliquid.sdk.utils.JSONUtil;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;
import org.jetbrains.annotations.NotNull;

public class WebsocketManager {
    private static final Logger LOG = Logger.getLogger(WebsocketManager.class.getName());
    private final String baseUrl;
    private final String wsUrl;
    private String probeUrl;
    private boolean probeDisabled = false;
    private final OkHttpClient client;
    private WebSocket webSocket;
    private volatile boolean stopped = false;
    private volatile boolean connected = false;
    private int reconnectAttempts = 0;
    private long backoffMs;
    private long initialBackoffMs = this.backoffMs = 1000L;
    private final long maxBackoffMs = 30000L;
    private long configMaxBackoffMs = 30000L;
    private volatile ScheduledFuture<?> reconnectFuture;
    private volatile boolean networkAvailable = true;
    private int networkCheckIntervalSeconds = 5;
    private volatile ScheduledFuture<?> networkMonitorFuture;
    private final OkHttpClient networkClient;
    private final Map<String, List<ActiveSubscription>> subscriptions = new ConcurrentHashMap<String, List<ActiveSubscription>>();
    private final Map<String, String> identifierCache = new ConcurrentHashMap<String, String>();
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    private final List<ConnectionListener> connectionListeners = Collections.synchronizedList(new ArrayList());
    private final List<CallbackErrorListener> callbackErrorListeners = Collections.synchronizedList(new ArrayList());

    public WebsocketManager(String baseUrl) {
        this.baseUrl = baseUrl;
        String scheme = baseUrl.startsWith("https") ? "wss" : "ws";
        String tail = baseUrl.replaceFirst("https?", "");
        this.wsUrl = scheme + tail + "/ws";
        this.probeUrl = null;
        this.client = new OkHttpClient.Builder().pingInterval(Duration.ofSeconds(20L)).readTimeout(Duration.ofSeconds(0L)).build();
        this.networkClient = new OkHttpClient.Builder().connectTimeout(Duration.ofSeconds(3L)).readTimeout(Duration.ofSeconds(3L)).callTimeout(Duration.ofSeconds(5L)).build();
        this.connect();
        this.startPing();
    }

    private void connect() {
        this.notifyConnecting();
        Request request = new Request.Builder().url(this.wsUrl).build();
        this.webSocket = this.client.newWebSocket(request, new WebSocketListener(){

            public void onOpen(@NotNull WebSocket webSocket, @NotNull Response response) {
                WebsocketManager.this.connected = true;
                WebsocketManager.this.reconnectAttempts = 0;
                WebsocketManager.this.backoffMs = WebsocketManager.this.initialBackoffMs;
                WebsocketManager.this.stopNetworkMonitor();
                WebsocketManager.this.notifyConnected();
                for (List<ActiveSubscription> list : WebsocketManager.this.subscriptions.values()) {
                    for (ActiveSubscription sub : list) {
                        WebsocketManager.this.sendSubscribe(sub.subscription);
                    }
                }
            }

            public void onMessage(@NotNull WebSocket webSocket, @NotNull String text) {
                try {
                    JsonNode msg = JSONUtil.readTree(text);
                    String identifier = WebsocketManager.this.wsMsgToIdentifier(msg);
                    if (identifier != null && WebsocketManager.this.subscriptions.containsKey(identifier)) {
                        for (ActiveSubscription sub : WebsocketManager.this.subscriptions.get(identifier)) {
                            try {
                                sub.callback.onMessage(msg);
                            }
                            catch (Exception cbEx) {
                                LOG.log(Level.WARNING, "WebSocket callback exception, identifier=" + identifier, cbEx);
                                WebsocketManager.this.notifyCallbackError(identifier, msg, cbEx);
                            }
                        }
                    }
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }

            public void onMessage(@NotNull WebSocket webSocket, @NotNull ByteString bytes) {
                this.onMessage(webSocket, bytes.utf8());
            }

            public void onFailure(@NotNull WebSocket webSocket, @NotNull Throwable t, Response response) {
                WebsocketManager.this.connected = false;
                WebsocketManager.this.notifyDisconnected(-1, String.valueOf(t), t);
                if (!WebsocketManager.this.stopped) {
                    WebsocketManager.this.scheduleReconnect(t, null, null);
                }
            }

            public void onClosing(@NotNull WebSocket webSocket, int code, @NotNull String reason) {
                webSocket.close(code, reason);
            }

            public void onClosed(@NotNull WebSocket webSocket, int code, @NotNull String reason) {
                WebsocketManager.this.connected = false;
                WebsocketManager.this.notifyDisconnected(code, reason, null);
                if (!WebsocketManager.this.stopped) {
                    WebsocketManager.this.scheduleReconnect(null, code, reason);
                }
            }
        });
    }

    private void startPing() {
        this.scheduler.scheduleAtFixedRate(this::sendPing, 20L, 20L, TimeUnit.SECONDS);
    }

    private void sendPing() {
        if (this.webSocket != null && this.connected) {
            Map<String, String> payload = Map.of("method", "ping");
            try {
                this.webSocket.send(JSONUtil.writeValueAsString(payload));
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    public void stop() {
        this.stopped = true;
        this.cancelTask(this.reconnectFuture);
        this.cancelTask(this.networkMonitorFuture);
        if (this.webSocket != null) {
            try {
                this.webSocket.close(1000, "stop");
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.webSocket = null;
        }
        try {
            this.scheduler.shutdown();
            if (!this.scheduler.awaitTermination(2L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
        try {
            this.client.dispatcher().executorService().shutdown();
            this.client.connectionPool().evictAll();
        }
        catch (Exception exception) {
            // empty catch block
        }
        try {
            this.networkClient.dispatcher().executorService().shutdown();
            this.networkClient.connectionPool().evictAll();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private synchronized void scheduleReconnect(Throwable cause, Integer code, String reason) {
        if (this.stopped) {
            return;
        }
        if (this.webSocket != null) {
            try {
                this.webSocket.close(1001, "reconnect");
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.webSocket = null;
        }
        long nextDelay = this.backoffMs + (long)(Math.random() * 250.0);
        this.notifyReconnecting(this.reconnectAttempts + 1, nextDelay);
        this.cancelTask(this.reconnectFuture);
        this.reconnectFuture = this.scheduler.schedule(() -> {
            if (!this.stopped) {
                this.connect();
            }
        }, nextDelay, TimeUnit.MILLISECONDS);
        ++this.reconnectAttempts;
        this.backoffMs = Math.min(Math.min(30000L, this.configMaxBackoffMs), this.backoffMs * 2L);
        this.startNetworkMonitor();
    }

    private synchronized void startNetworkMonitor() {
        if (this.networkMonitorFuture != null && !this.networkMonitorFuture.isCancelled()) {
            return;
        }
        this.networkMonitorFuture = this.scheduler.scheduleWithFixedDelay(() -> {
            boolean ok = this.isNetworkAvailable();
            if (ok) {
                if (!this.networkAvailable) {
                    this.networkAvailable = true;
                    this.notifyNetworkAvailable();
                }
                if (!this.connected && !this.stopped) {
                    this.backoffMs = this.initialBackoffMs;
                    this.reconnectAttempts = 0;
                    this.cancelTask(this.reconnectFuture);
                    this.notifyReconnecting(1, 0L);
                    this.reconnectFuture = this.scheduler.schedule(this::connect, 0L, TimeUnit.MILLISECONDS);
                }
            } else if (this.networkAvailable) {
                this.networkAvailable = false;
                this.notifyNetworkUnavailable();
            }
        }, 0L, this.networkCheckIntervalSeconds, TimeUnit.SECONDS);
    }

    private synchronized void stopNetworkMonitor() {
        if (this.networkMonitorFuture != null) {
            this.networkMonitorFuture.cancel(false);
            this.networkMonitorFuture = null;
        }
        this.networkAvailable = true;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean isNetworkAvailable() {
        if (this.probeDisabled) {
            return true;
        }
        String url = this.probeUrl != null ? this.probeUrl : this.baseUrl;
        int maxRetries = 2;
        long retryDelayMs = 100L;
        int attempt = 0;
        while (attempt < maxRetries) {
            try {
                Request req = new Request.Builder().url(url).head().build();
                try (Response resp = this.networkClient.newCall(req).execute();){
                    if (resp.code() < 400) {
                        boolean bl = true;
                        return bl;
                    }
                }
            }
            catch (Exception e) {
                if (attempt == maxRetries - 1) {
                    LOG.log(Level.FINE, "Network detection failed, retried " + maxRetries + " times", e);
                    return false;
                }
                try {
                    Thread.sleep(retryDelayMs);
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    return false;
                }
            }
            ++attempt;
        }
        return false;
    }

    public void setNetworkProbeUrl(String url) {
        this.probeUrl = url;
    }

    public void setNetworkProbeDisabled(boolean disabled) {
        this.probeDisabled = disabled;
    }

    public void addConnectionListener(ConnectionListener l) {
        if (l != null) {
            this.connectionListeners.add(l);
        }
    }

    public void removeConnectionListener(ConnectionListener l) {
        if (l != null) {
            this.connectionListeners.remove(l);
        }
    }

    public void addCallbackErrorListener(CallbackErrorListener l) {
        if (l != null) {
            this.callbackErrorListeners.add(l);
        }
    }

    public void removeCallbackErrorListener(CallbackErrorListener l) {
        if (l != null) {
            this.callbackErrorListeners.remove(l);
        }
    }

    public void setNetworkCheckIntervalSeconds(int seconds) {
        this.networkCheckIntervalSeconds = Math.max(1, seconds);
    }

    public void setReconnectBackoffMs(long initialMs, long maxMs) {
        long init = Math.max(100L, initialMs);
        long max = Math.max(init, maxMs);
        this.initialBackoffMs = init;
        this.backoffMs = init;
        this.configMaxBackoffMs = Math.min(30000L, max);
    }

    private void cancelTask(ScheduledFuture<?> future) {
        if (future != null && !future.isCancelled()) {
            future.cancel(false);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> void notifyListeners(List<T> listeners, Consumer<T> action) {
        List<T> list = listeners;
        synchronized (list) {
            for (T listener : listeners) {
                try {
                    action.accept(listener);
                }
                catch (Exception exception) {}
            }
        }
    }

    private void notifyConnecting() {
        this.notifyListeners(this.connectionListeners, l -> l.onConnecting(this.wsUrl));
    }

    private void notifyConnected() {
        this.notifyListeners(this.connectionListeners, l -> l.onConnected(this.wsUrl));
    }

    private void notifyDisconnected(int code, String reason, Throwable cause) {
        this.notifyListeners(this.connectionListeners, l -> l.onDisconnected(this.wsUrl, code, reason, cause));
    }

    private void notifyReconnecting(int attempt, long nextDelayMs) {
        this.notifyListeners(this.connectionListeners, l -> l.onReconnecting(this.wsUrl, attempt, nextDelayMs));
    }

    private void notifyReconnectFailed(int attempted, Throwable lastError) {
        this.notifyListeners(this.connectionListeners, l -> l.onReconnectFailed(this.wsUrl, attempted, lastError));
    }

    private void notifyNetworkUnavailable() {
        this.notifyListeners(this.connectionListeners, l -> l.onNetworkUnavailable(this.wsUrl));
    }

    private void notifyNetworkAvailable() {
        this.notifyListeners(this.connectionListeners, l -> l.onNetworkAvailable(this.wsUrl));
    }

    private void notifyCallbackError(String identifier, JsonNode msg, Throwable error) {
        this.notifyListeners(this.callbackErrorListeners, l -> l.onCallbackError(this.wsUrl, identifier, msg, error));
    }

    public void subscribe(Subscription subscription, MessageCallback callback) {
        if (this.stopped) {
            throw new IllegalStateException("WebsocketManager has been stopped, cannot subscribe");
        }
        if (subscription == null) {
            throw new IllegalArgumentException("subscription cannot be null");
        }
        if (callback == null) {
            throw new IllegalArgumentException("callback cannot be null");
        }
        JsonNode jsonNode = JSONUtil.convertValue((Object)subscription, JsonNode.class);
        this.subscribe(jsonNode, callback);
    }

    public void subscribe(JsonNode subscription, MessageCallback callback) {
        if (this.stopped) {
            throw new IllegalStateException("WebsocketManager has been stopped, cannot subscribe");
        }
        if (subscription == null) {
            throw new IllegalArgumentException("subscription cannot be null");
        }
        if (callback == null) {
            throw new IllegalArgumentException("callback cannot be null");
        }
        String identifier = this.subscriptionToIdentifier(subscription);
        List list = this.subscriptions.computeIfAbsent(identifier, k -> new CopyOnWriteArrayList());
        for (ActiveSubscription s : list) {
            if (!s.subscription.equals((Object)subscription)) continue;
            return;
        }
        list.add(new ActiveSubscription(subscription, callback));
        this.sendSubscribe(subscription);
    }

    private void sendSubscribe(JsonNode subscription) {
        if (this.webSocket == null || !this.connected) {
            return;
        }
        LinkedHashMap<String, String> payload = new LinkedHashMap<String, String>();
        payload.put("method", "subscribe");
        payload.put("subscription", (String)subscription);
        try {
            this.webSocket.send(JSONUtil.writeValueAsString(payload));
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public void unsubscribe(Subscription subscription) {
        if (subscription == null) {
            throw new IllegalArgumentException("subscription cannot be null");
        }
        JsonNode jsonNode = JSONUtil.convertValue((Object)subscription, JsonNode.class);
        this.unsubscribe(jsonNode);
    }

    public void unsubscribe(JsonNode subscription) {
        if (subscription == null) {
            throw new IllegalArgumentException("subscription cannot be null");
        }
        String identifier = this.subscriptionToIdentifier(subscription);
        List<ActiveSubscription> list = this.subscriptions.get(identifier);
        if (list != null) {
            list.removeIf(s -> s.subscription.equals((Object)subscription));
            if (list.isEmpty()) {
                this.subscriptions.remove(identifier);
            }
        }
        if (this.webSocket == null || !this.connected) {
            return;
        }
        LinkedHashMap<String, String> payload = new LinkedHashMap<String, String>();
        payload.put("method", "unsubscribe");
        payload.put("subscription", (String)subscription);
        try {
            this.webSocket.send(JSONUtil.writeValueAsString(payload));
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    String subscriptionToIdentifier(JsonNode subscription) {
        String type;
        if (subscription == null || !subscription.has("type")) {
            return "unknown";
        }
        switch (type = subscription.get("type").asText()) {
            case "allMids": 
            case "userEvents": 
            case "orderUpdates": {
                return type;
            }
            case "l2Book": {
                JsonNode coinNode = subscription.get("coin");
                String coinKey = this.extractCoinIdentifier(coinNode);
                return this.buildCachedIdentifier(type, coinKey);
            }
            case "trades": 
            case "bbo": 
            case "activeAssetCtx": {
                JsonNode coinNode = subscription.get("coin");
                String coinKey = this.extractCoinIdentifier(coinNode);
                return this.buildCachedIdentifier(type, coinKey);
            }
            case "candle": {
                String interval;
                JsonNode coinNode = subscription.get("coin");
                JsonNode iNode = subscription.get("interval");
                String coinKey = this.extractCoinIdentifier(coinNode);
                String string = interval = iNode == null ? null : iNode.asText();
                if (coinKey != null && interval != null) {
                    return this.buildCachedIdentifier(type, coinKey + "," + interval);
                }
                return type;
            }
            case "userFills": 
            case "userFundings": 
            case "userNonFundingLedgerUpdates": 
            case "webData2": {
                JsonNode userNode = subscription.get("user");
                String user = userNode == null ? null : userNode.asText().toLowerCase(Locale.ROOT);
                return this.buildCachedIdentifier(type, user);
            }
            case "activeAssetData": {
                String user;
                JsonNode coinNode = subscription.get("coin");
                JsonNode userNode = subscription.get("user");
                String coinKey = this.extractCoinIdentifier(coinNode);
                String string = user = userNode == null ? null : userNode.asText().toLowerCase(Locale.ROOT);
                if (coinKey != null && user != null) {
                    return this.buildCachedIdentifier(type, coinKey + "," + user);
                }
                return type;
            }
        }
        return type;
    }

    private String extractCoinIdentifier(JsonNode coinNode) {
        if (coinNode == null) {
            return null;
        }
        return coinNode.isNumber() ? String.valueOf(coinNode.asInt()) : coinNode.asText().toLowerCase(Locale.ROOT);
    }

    private String buildCachedIdentifier(String type, String suffix) {
        if (suffix == null) {
            return type;
        }
        String cacheKey = type + "|" + suffix;
        return this.identifierCache.computeIfAbsent(cacheKey, k -> type + ":" + suffix);
    }

    String wsMsgToIdentifier(JsonNode msg) {
        if (msg == null || !msg.has("channel")) {
            return null;
        }
        JsonNode channelNode = msg.get("channel");
        String type = null;
        if (channelNode.isTextual()) {
            type = channelNode.asText();
        } else if (channelNode.isObject() && channelNode.has("type")) {
            type = channelNode.get("type").asText();
        }
        if (type == null) {
            return null;
        }
        switch (type) {
            case "pong": 
            case "allMids": 
            case "userEvents": 
            case "orderUpdates": {
                return type;
            }
            case "l2Book": {
                JsonNode coinNode = msg.path("data").path("coin");
                String coinKey = this.extractCoinIdentifier(coinNode);
                return this.buildCachedIdentifier(type, coinKey);
            }
            case "trades": {
                JsonNode first;
                JsonNode coinNode;
                JsonNode trades = msg.get("data");
                String coinKey = null;
                if (trades != null && trades.isArray() && !trades.isEmpty() && (coinNode = (first = trades.get(0)).get("coin")) != null) {
                    coinKey = this.extractCoinIdentifier(coinNode);
                }
                return this.buildCachedIdentifier(type, coinKey);
            }
            case "candle": {
                JsonNode data = msg.get("data");
                if (data != null) {
                    String s = data.path("s").asText(null);
                    String i = data.path("i").asText(null);
                    if (s != null && i != null) {
                        return this.buildCachedIdentifier(type, s.toLowerCase(Locale.ROOT) + "," + i);
                    }
                }
                return type;
            }
            case "bbo": {
                JsonNode coinNode = msg.path("data").path("coin");
                String coinKey = this.extractCoinIdentifier(coinNode);
                return this.buildCachedIdentifier(type, coinKey);
            }
            case "userFills": 
            case "userFundings": 
            case "userNonFundingLedgerUpdates": 
            case "webData2": {
                JsonNode userNode = msg.path("data").path("user");
                String user = userNode != null && userNode.isTextual() ? userNode.asText().toLowerCase(Locale.ROOT) : null;
                return this.buildCachedIdentifier(type, user);
            }
            case "activeAssetCtx": 
            case "activeSpotAssetCtx": {
                JsonNode coinNode = msg.path("data").path("coin");
                String coinKey = this.extractCoinIdentifier(coinNode);
                return type.equals("activeSpotAssetCtx") ? this.buildCachedIdentifier("activeAssetCtx", coinKey != null ? coinKey : "unknown") : this.buildCachedIdentifier(type, coinKey);
            }
            case "activeAssetData": {
                JsonNode data = msg.get("data");
                if (data != null) {
                    String user;
                    JsonNode coinNode = data.get("coin");
                    JsonNode userNode = data.get("user");
                    String coinKey = this.extractCoinIdentifier(coinNode);
                    String string = user = userNode != null && userNode.isTextual() ? userNode.asText().toLowerCase(Locale.ROOT) : null;
                    if (coinKey != null && user != null) {
                        return this.buildCachedIdentifier(type, coinKey + "," + user);
                    }
                }
                return type;
            }
        }
        return type;
    }

    public Map<String, List<ActiveSubscription>> getSubscriptions() {
        HashMap<String, List<ActiveSubscription>> copy = new HashMap<String, List<ActiveSubscription>>();
        for (Map.Entry<String, List<ActiveSubscription>> entry : this.subscriptions.entrySet()) {
            copy.put(entry.getKey(), new ArrayList(entry.getValue()));
        }
        return copy;
    }

    public List<ActiveSubscription> getSubscriptionsByIdentifier(String identifier) {
        List<ActiveSubscription> list = this.subscriptions.get(identifier);
        return list != null ? new ArrayList<ActiveSubscription>(list) : Collections.emptyList();
    }

    public boolean hasSubscriptions() {
        return !this.subscriptions.isEmpty();
    }

    public int getSubscriptionCount() {
        return this.subscriptions.size();
    }

    public static interface MessageCallback {
        public void onMessage(JsonNode var1);
    }

    public static class ActiveSubscription {
        public final JsonNode subscription;
        public final MessageCallback callback;

        public ActiveSubscription(JsonNode s, MessageCallback c) {
            this.subscription = s;
            this.callback = c;
        }

        public JsonNode getSubscription() {
            return this.subscription;
        }
    }

    public static interface CallbackErrorListener {
        public void onCallbackError(String var1, String var2, JsonNode var3, Throwable var4);
    }

    public static interface ConnectionListener {
        public void onConnecting(String var1);

        public void onConnected(String var1);

        public void onDisconnected(String var1, int var2, String var3, Throwable var4);

        public void onReconnecting(String var1, int var2, long var3);

        public void onReconnectFailed(String var1, int var2, Throwable var3);

        public void onNetworkUnavailable(String var1);

        public void onNetworkAvailable(String var1);
    }
}

