package io.github.kaststream.api.v2;

import io.github.kaststream.api.v2.facade.*;
import java.io.Serializable;
import java.util.Map;
import org.apache.avro.Schema;

public interface CoProcessor extends Serializable {

  /**
   * Gets the data type (as a {@link Schema}) produced by this function or input format.
   *
   * @param schema1 the input row schema from stream 1
   * @param schema2 the input row schema from stream 2
   * @return The data type produced by this function or input format.
   */
  public Schema getProducedType(final Schema schema1, final Schema schema2);

  /**
   * Declare the data type information of the sides out data produced by your function.
   *
   * @return the sides output type information
   */
  public default Map<String, Schema> getSidesOutProducedType(
      final Schema schema1, final Schema schema2) {
    return null;
  }

  /**
   * This method is called for each element in the first of the connected streams.
   *
   * <p>This function can output zero or more elements using the {@link KastCollector} parameter.
   *
   * @param kRow The stream element
   * @param ctx A {@link KastCoProcessorContext} that allows querying the timestamp of the element.
   *     The context is only valid during the invocation of this method, do not store it.
   * @param out The collector to emit resulting elements to
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public void processElement1(
      final KastRow kRow,
      final KastCoProcessorContext ctx,
      final KastCollector out,
      KastRowBuilder kastRowBuilder)
      throws Exception;

  /**
   * This method is called for each element in the second of the connected streams.
   *
   * <p>This function can output zero or more elements using the {@link KastCollector} parameter and
   * also update internal state or set timers using the {@link KastCoProcessorContext} parameter.
   *
   * @param kRow The stream element
   * @param ctx A {@link KastCoProcessorContext} that allows querying the timestamp of the element.
   *     The context is only valid during the invocation of this method, do not store it.
   * @param out The collector to emit resulting elements to
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public void processElement2(
      final KastRow kRow,
      final KastCoProcessorContext ctx,
      final KastCollector out,
      KastRowBuilder kastRowBuilder)
      throws Exception;

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema1 the input row schema from stream 1
   * @param schema2 the input row schema from stream 2
   * @param sidesOutputTag Side output typeinfo and tag dict if any
   * @param rtCtx a runtime context that allow you for example to manage state
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration,
      final Schema schema1,
      final Schema schema2,
      final Map<String, KastOutputTag> sidesOutputTag,
      final KastRuntimeContext rtCtx)
      throws Exception
      {}
  ;

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema1 the input row schema from stream 1
   * @param schema2 the input row schema from stream 2
   * @param sidesOutputTag Side output typeinfo and tag dict if any
   * @param rtCtx a runtime context that allow you for example to manage state
   * @param meta the udf configuration parameters
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration,
      final Schema schema1,
      final Schema schema2,
      final Map<String, KastOutputTag> sidesOutputTag,
      final KastRuntimeContext rtCtx,
      final Map<String, Object> meta)
      throws Exception {
    throw new UnsupportedOperationException("open with meta not implemented");
  }
  ;

  /**
   * Tear-down method for the user code. It is called after the last call to the main working
   * methods. For functions that are part of an iteration, this method will be invoked after each
   * iteration superstep.
   *
   * <p>This method can be used for clean up work.
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void close() throws Exception {}
  ;
}
