package pluginloader.gradle

import com.jcraft.jsch.ChannelSftp
import com.jcraft.jsch.JSch
import com.jcraft.jsch.agentproxy.ConnectorFactory
import com.jcraft.jsch.agentproxy.RemoteIdentityRepository
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.file.DuplicatesStrategy
import org.gradle.api.plugins.JavaPluginExtension
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.jvm.tasks.Jar
import org.gradle.plugins.signing.SigningExtension
import java.io.File
import java.nio.file.Files

private val kotlinVersion = "1.6.10"
private val kotlinSerializationVersion = "1.3.1"
private val pluVersion = "1.11.10"
private val affectedPlugins = setOf("ksyml", "configs", "text", "provide", "cmdexec")//gradle broken these plugins on first upload, thanks fucking gradle

open class Config(private val project: Project){
    internal var prefix = "PLU"
    internal var expand: String? = null

    fun expand(plugin: String){
        expand = plugin
    }

    fun central(vararg plugins: String){
        plugins.forEach{project.dependencies.add("dependency",
            "io.github.pluginloader:${if(!it.contains(':')){
                if(it in affectedPlugins) "$it:1.0.1" else "$it:1.0.0"
            } else it}")}
    }

    fun plu(vararg plugins: String){
        plugins.forEach{project.dependencies.add("dependency", "pluginloader:${if(!it.contains(':')) "$it:1.0.0" else it}")}
    }

    fun impl(vararg dep: String){
        dep.forEach{project.dependencies.add("implementation", it)}
    }

    fun compileOnly(vararg dep: String){
        dep.forEach{project.dependencies.add("compileOnly", it)}
    }

    fun includeDep(){
        project.tasks.withType(org.gradle.api.tasks.bundling.Jar::class.java){
            it.duplicatesStrategy = DuplicatesStrategy.INCLUDE
            it.from(project.configurations.getByName("runtimeClasspath").map{map -> if(map.isDirectory) map else project.zipTree(map)})
        }
    }

    fun paper(version: String){
        project.repositories.add(project.repositories.maven{it.url = project.uri("https://papermc.io/repo/repository/maven-public/")})
        project.dependencies.add("compileOnly", "io.github.pluginloader:bukkit-api:$pluVersion")
        if(version == "1.12.2"){
            project.dependencies.add("compileOnly", "com.destroystokyo.paper:paper-api:$version-R0.1-SNAPSHOT")
        }else{//1.13+?
            project.dependencies.add("compileOnly", "io.papermc.paper:paper-api:$version-R0.1-SNAPSHOT")
        }
    }

    fun public() {
        if (getenv(project, "MAVEN_CENTRAL_USER") == null
            || getenv(project, "MAVEN_CENTRAL_PASSWORD") == null) return

        val publishing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("publishing") as org.gradle.api.publish.PublishingExtension

        val java = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("java") as JavaPluginExtension

        java.withSourcesJar()
        java.withJavadocJar()

        publishing.publications.create("maven", MavenPublication::class.java){
            it.from(project.components.getByName("java"))
            it.pom {m ->
                m.name.set("Pluginloader")
                m.description.set("One of many Pluginloader plugins")
                m.url.set("https://github.com/pluginloader/mono")

                m.licenses {l ->
                    l.license {lic ->
                        lic.name.set("MIT")
                        lic.url.set("https://mit-license.org/")
                    }
                }
                m.developers {d ->
                    d.developer {dev ->
                        dev.id.set("6oogle")
                        dev.name.set("6oogle")
                        dev.email.set("mail@example.com")
                    }
                }
                m.scm {scm ->
                    scm.connection.set("scm:git:https://github.com/pluginloader/mono.git")
                    scm.developerConnection.set("scm:git:https://github.com/pluginloader/mono.git")
                    scm.url.set("https://github.com/pluginloader/mono")
                }
            }
        }

        publishing.repositories {
            it.maven {repo ->
                repo.url = project.uri("https://s01.oss.sonatype.org/service/local/staging/deploy/maven2/")
                repo.credentials {cre ->
                    cre.username = getenv(project, "MAVEN_CENTRAL_USER")!!
                    cre.password = getenv(project, "MAVEN_CENTRAL_PASSWORD")!!
                }
            }
        }

        val signing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("signing") as SigningExtension
        signing.useGpgCmd()
        signing.sign(publishing.publications.getByName("maven"))
    }

    fun private(prefix: String = "PLU") {
        if (getenv(project, "${prefix}_NEXUS_URL") == null){
            error("Can't add private repo, env ${prefix}_NEXUS_URL not found")
        }
        if (getenv(project, "${prefix}_NEXUS_PUSH_USER") == null){
            error("Can't add private repo, env ${prefix}_NEXUS_PUSH_USER not found")
        }
        if (getenv(project, "${prefix}_NEXUS_PUSH_PASSWORD") == null){
                error("Can't add private repo, ${prefix}_NEXUS_PUSH_PASSWORD not found")
        }
        this.prefix = prefix
        project.repositories.maven {
            it.url = project.uri(getenv(project, "${prefix}_NEXUS_URL")!!)

            it.credentials { password ->
                password.username = getenv(project, "${prefix}_NEXUS_PUSH_USER")!!
                password.password = getenv(project, "${prefix}_NEXUS_PUSH_PASSWORD")!!
            }
        }
        val publishing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("publishing") as org.gradle.api.publish.PublishingExtension
        publishing.publications.create("java", MavenPublication::class.java){
            it.from(project.components.getByName("java"))
        }
        publishing.repositories.maven {
            it.url = project.uri(getenv(project, "${prefix}_NEXUS_URL")!!)

            it.credentials { password ->
                password.username = getenv(project, "${prefix}_NEXUS_PUSH_USER")!!
                password.password = getenv(project, "${prefix}_NEXUS_PUSH_PASSWORD")!!
            }
        }
    }
}

private fun getenv(project: Project, env: String): String?{
    if(project.hasProperty(env))return project.property(env).toString()
    return System.getenv(env)
}

class GradlePlugin: Plugin<Project> {
    override fun apply(project: Project) {
        val config = Config(project)
        project.extensions.add("plu", config)
        val pluProjectFile = project.file("plu_project.txt")
        val pluProjectPath = pluProjectFile.toPath()
        if(Files.exists(pluProjectPath)){
            val read = pluProjectFile.readText().split("\n")[0]//template
            val name = project.file(".").name//plugin name or template
            if(!name.contains("template")){
                Files.delete(pluProjectPath)
                val src = project.file("src/main/kotlin/Plugin.kt")
                val srcPath = src.toPath()
                if(Files.exists(srcPath)) src.writeText(src.readText().replace(read, name))
                val settings = project.file("./settings.gradle.kts")
                val settingsPath = settings.toPath()
                if(Files.exists(settingsPath)) settings.writeText(settings.readText().replace(read, name))
            }
        }

        if(!project.pluginManager.hasPlugin("org.jetbrains.kotlin.jvm")){
            project.pluginManager.apply("org.jetbrains.kotlin.jvm")
        }
        if(!project.pluginManager.hasPlugin("org.jetbrains.kotlin.plugin.serialization")){
            project.pluginManager.apply("org.jetbrains.kotlin.plugin.serialization")
        }
        if(!project.pluginManager.hasPlugin("signing")) {
            project.pluginManager.apply("signing")
        }

        project.tasks.getByName("jar").doFirst{_ ->
            val dir = project.file("build/classes/kotlin/main/pluginloader")
            if(dir.exists())dir.deleteRecursively()
            project.configurations.getByName("dependency").allDependencies.forEach{
                val path = project.file("build/classes/kotlin/main/pluginloader/${it.name}.dependency").toPath()
                Files.createDirectories(path.parent)
                Files.createFile(path)
                Files.write(path, "${it.group}:${it.name}:${it.version}".toByteArray())
            }
            project.configurations.getByName("mavenDependency").allDependencies.forEach{
                val path = project.file("build/classes/kotlin/main/pluginloader/${it.group};${it.name};${it.version}.mavenDependency").toPath()
                Files.createDirectories(path.parent)
                Files.createFile(path)
            }
            config.expand?.apply {
                val path = project.file("build/classes/kotlin/main/pluginloader/${this}.expand").toPath()
                Files.createDirectories(path.parent)
                Files.createFile(path)
            }
        }

        //if(project.pluginManager.hasPlugin("org.hidetake.ssh").not()){
        //    project.pluginManager.apply(org.hidetake.gradle.ssh.plugin.SshPlugin::class.java)
        //}
        project.task("upload"){
            it.dependsOn("build")
            it.doLast{
                val host = getenv(project, "${config.prefix}_PUSH_HOST") ?:
                    error("Can't find property ${config.prefix}_PUSH_HOST, example: 'example.com', also can set ${config.prefix}_PUSH_PORT, example: '22'")
                val user = getenv(project, "${config.prefix}_PUSH_USER") ?: error("Can't find property ${config.prefix}_PUSH_USER, example: 'user'")
                val port = getenv(project, "${config.prefix}_PUSH_PORT")?.toInt() ?: 22
                val uploadFrom = "build/libs/${project.name}.jar"
                val uploadTo = project.properties["p"] ?: getenv(project, "${config.prefix}_PUSH_DIR") ?:
                    error("Can't find property ${config.prefix}_PUSH_DIR, example: 'dir/dir', also can use 'gradle upload -Pp=path/to/dir'")
                val jsch = JSch()
                jsch.setKnownHosts("${System.getProperty("user.home")}${File.separator}.ssh${File.separator}known_hosts")
                jsch.setIdentityRepository(RemoteIdentityRepository(ConnectorFactory.getDefault().createConnector()))
                val session = jsch.getSession(user, host, port)
                session.connect()
                val sftp = session.openChannel("sftp") as ChannelSftp
                sftp.connect()
                sftp.put(project.file(uploadFrom).absolutePath, uploadTo.toString())
                sftp.disconnect()
                session.disconnect()
            }
        }

        project.pluginManager.apply("maven-publish")

        val jarTask = project.tasks.getByPath("jar")
        if(jarTask is Jar) jarTask.archiveFileName.set("${project.name}.jar")

        project.repositories.add(project.repositories.mavenCentral())

        val dependency = project.configurations.create("dependency")
        dependency.isTransitive = false
        project.configurations.getByName("compileClasspath").extendsFrom(dependency)

        project.dependencies.add("compileOnly", "org.jetbrains.kotlin:kotlin-stdlib:$kotlinVersion")
        project.dependencies.add("compileOnly", "org.jetbrains.kotlinx:kotlinx-serialization-core:$kotlinSerializationVersion")
        project.dependencies.add("compileOnly", "org.jetbrains.kotlinx:kotlinx-serialization-json:$kotlinSerializationVersion")
        project.dependencies.add("compileOnly", "io.github.pluginloader:api:$pluVersion")

        val mvnDependency = project.configurations.create("mavenDependency")
        mvnDependency.isTransitive = false
        project.configurations.getByName("compileClasspath").extendsFrom(mvnDependency)
    }
}