package armyc2.c5isr.renderer.utilities;

import java.util.ArrayList;
import java.util.Map;

/**
 * Modifier Constants to be used as keys in the modifiers map when calling {@link armyc2.c5isr.renderer.MilStdIconRenderer#RenderIcon(String, Map, Map)}
 * or {@link armyc2.c5isr.web.render.WebRenderer#RenderSymbol(String, String, String, String, String, String, double, String, Map, Map, int)}
 */
public class Modifiers {
    /**
     * <pre>
     * Symbol Icon
     * The innermost part of a symbol, comprised of an icon and optional modifiers, that represents a joint military object (see 5.3.4).
     * Format: Graphic
     * Symbol Set: All
     * Remarks: Determined by SIDC positions 11-20.
     * </pre>
     */
    public static final String A_SYMBOL_ICON = "A_SYMBOL_ICON";
    /**
     * <pre>
     * Echelon
     * An amplifier in a unit symbol that identifies command level (see 5.3.7.1 and table VII).
     * Format: Graphic
     * Symbol Set: All
     * Remarks: Determined by SIDC positions 9-10.
     * </pre>
     */
    public static final String B_ECHELON = "B_ECHELON";
    /**
     * <pre>
     * Quantity
     * An amplifier in an equipment symbol that identifies the number of items present.
     * Examples include:
     * 350 Beds 50 Gallons
     * Format:
     * Alphanumeric - {1,19}
     * [#########] [XXXXXXXXXX]
     * Symbol Set: 10, 11, 15, 25, 27, 60
     * Remarks: Two-part composite field.
     * Where # is the numeric value [1-999999999], and X is the unit of measure.
     * Note: There should be a space between the numeric and the unit of measure values.
     * </pre>
     */
    public static final String C_QUANTITY = "C_QUANTITY";
    /**
     * <pre>
     * Task Force Indicator
     * An amplifier that identifies a unit or activities symbol as a task force (see 5.3.7.2 and figure 14).
     * Format: Graphic
     * Symbol Set: 10
     * Remarks: Determined by SIDC position 8.
     * </pre>
     */
    public static final String D_TASK_FORCE_INDICATOR = "D_TASK_FORCE_INDICATOR";

    /**
     * <pre>
     * Frame Shape Modifier
     * A graphic modifier that displays standard identity, battle dimension, or exercise
     * amplifying descriptors of an object (see 5.3.1 and table II).
     * Format: Graphic
     * Symbol Set: ALL BUT 25,45,46,47
     * Remarks: 2525C, not processed as a modifier in 2525D+
     * </pre>
     */
    public static final String E_FRAME_SHAPE_MODIFIER = "E_FRAME_SHAPE_MODIFIER";

    /**
     * <pre>
     * Reinforced or Reduced
     * An amplifier in a unit symbol that displays (+) for reinforced, (-) for reduced, (<u>+</u>) reinforced and reduced.
     * Format: Alphanumeric - {1,1}
     * Symbol Set: 10
     * Remarks:
     * </pre>
     */
    public static final String F_REINFORCED_REDUCED = "F_REINFORCED_REDUCED";
    /**
     * <pre>
     * Staff Comments
     * An amplifier for units, equipment and installations; content is implementation specific.
     * Format: Alphanumeric - {1,20}
     * Symbol Set: 01, 05, 10, 15, 20, 27, 30, 35, 40
     * Remarks:
     * </pre>
     */
    public static final String G_STAFF_COMMENTS = "G_STAFF_COMMENTS";
    /**
     * <pre>
     * Additional Information
     * An amplifier for units, equipment and installations; content is implementation specific.
     * Format: Alphanumeric - {1,20}
     * Symbol Set: ALL
     * Remarks:
     * </pre>
     */
    public static final String H_ADDITIONAL_INFO_1 = "H_ADDITIONAL_INFO_1";

    /**
     * <pre>
     * Unlisted Point Information
     * An alphanumeric text amplifier used to provide an up to a three-character letter field acronym to describe a point that is not already listed.
     * Format: Alphanumeric - {1,3}
     * Symbol Set: 25
     * Remarks: Only used with Action Points (General) control measure. SIDC 130100.
     * </pre>
     */
    public static final String H1_ADDITIONAL_INFO_2 = "H1_ADDITIONAL_INFO_2";

    /**
     * A text modifier for tactical graphics; content is
     * implementation specific.
     * CM: P,L,A,N,B/C,R/N
     * Length: 20
     * @deprecated 2525C
     */
    public static final String H2_ADDITIONAL_INFO_3 = "H2_ADDITIONAL_INFO_3";

    /**
     * <pre>
     * Evaluation Rating
     * An amplifier that consists of a one-letter reliability rating and a one-number credibility rating. (See ATP 2-33.4)
     * Format: Alphanumeric - {2,2} [X][#]
     * Symbol Set: 10, 15, 20,27, 40
     * Remarks:
     *
     * Reliability Ratings:
     * A-completely reliable
     * B-usually reliable
     * C-fairly reliable
     * D-not usually reliable
     * E-unreliable
     * F-reliability cannot be judged
     *
     * Credibility Ratings:
     * 1-confirmed by other sources
     * 2-probably true
     * 3-possibly true
     * 4-doubtfully true
     * 5-improbable
     * 6-truth cannot be judged
     * </pre>
     */
    public static final String J_EVALUATION_RATING = "J_EVALUATION_RATING";
    /**
     * <pre>
     * Effectiveness
     * An amplifier for units and installations that indicates unit effectiveness or installation capability.
     * Format: Alphanumeric - {2,3}
     * Symbol Set: 10, 15, 27
     * Remarks:
     * List of Values:
     * FO - Fully Operational
     * SO - Substantially operational
     * MO - Marginally operational
     * NO - Not operational
     * UNK - Unknown
     * </pre>
     */
    public static final String K_COMBAT_EFFECTIVENESS = "K_COMBAT_EFFECTIVENESS";
    /**
     * <pre>
     * Signature Equipment
     * An amplifier for hostile equipment; "!" indicates detectable electronic signatures.
     * Format: Alphanumeric - {1,1}
     * Symbol Set: 15
     * Remarks: The amplifier displayed is the exclamation mark "!".
     * </pre>
     */
    public static final String L_SIGNATURE_EQUIP = "L_SIGNATURE_EQUIP";
    /**
     * <pre>
     * Higher Formation
     * An amplifier for units that indicates number or title of higher echelon command (corps are designated by Roman numerals).
     * Format: Alphanumeric - {1,21}
     * Symbol Set: 10
     * Remarks: The amplifier displayed is the exclamation mark "!".
     * </pre>
     */
    public static final String M_HIGHER_FORMATION = "M_HIGHER_FORMATION";
    /**
     * <pre>
     * Hostile (Enemy)
     * An amplifier for equipment; letters "ENY" denote hostile symbols.
     * Format: Alphanumeric - {3,3}
     * Symbol Set: 15,25
     * Remarks: Determined by SIDC position 4.
     * Note: This amplifier must be used when displaying enemy/hostile control measures on monochromatic displays
     * </pre>
     */
    public static final String N_HOSTILE = "N_HOSTILE";
    /**
     * <pre>
     * IFF/SIF/AIS
     * An amplifier displaying IFF/SIF/AIS Identification modes and codes.
     * Format:
     * <br></br><img src="data:image/png;base64,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" alt="" />
     * Symbol Set: 01, 10, 15, 27, 30, 35
     * Remarks: 'MODE' is not required when displaying.
     * </pre>
     */
    public static final String P_IFF_SIF_AIS = "P_IFF_SIF_AIS";
    /**
     * <pre>
     * Direction of Movement Indicator
     * An amplifier consisting of a line with arrow that identifies the direction of movement or intended movement of an object (see 5.3.7.9 and figure 14).
     * Format: Graphic (Alphanumeric for exchange only)
     * Symbol Set: 10, 15, 20, 25, 27
     * Remarks:
     * Renderer Assumes a value in degrees with no text
     * MilStd:
     * <br></br><img src="data:image/png;base64,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" alt="" />
     *
     * </pre>
     */
    public static final String Q_DIRECTION_OF_MOVEMENT = "Q_DIRECTION_OF_MOVEMENT";
    /**
     * <pre>
     * Mobility Indicator
     * An amplifier that depicts the mobility of an object not intrinsic to the entity itself (see 5.3.7.9, figure 14 and table IX).
     * Format: Graphic
     * Symbol Set: 10
     * Remarks: Determined by SIDC positions 9-10
     * </pre>
     */
    public static final String R_MOBILITY_INDICATOR = "R_MOBILITY_INDICATOR";
    /**
     * <pre>
     * SIGINT Mobility Indicator
     * An amplifier that indicates the mobility of a SIGINT unit.
     * Format: Alphanumeric - {1,1}
     * Symbol Set: 50, 51, 52, 53, 54
     * Remarks:
     * List of Values:
     * M = Mobile,
     * S = Static
     * U = Uncertain.
     * </pre>
     */
    public static final String R2_SIGNIT_MOBILITY_INDICATOR = "R2_SIGNIT_MOBILITY_INDICATOR";
    /**
     * <pre>
     * Headquarters Staff Indicator
     * An amplifier for units, installations and activities that identifies them as a headquarters (see figure 14 and figure 13).
     * Format: Graphic
     * Symbol Set: 10, 20, 40
     * Remarks: Determined by SIDC position 8.
     * </pre>
     */
    public static final String S_HQ_STAFF_INDICATOR = "S_HQ_STAFF_INDICATOR";
    /**
     * <pre>
     * Offset Location Indicator
     * An amplifier used to indicate the offset or precise location of a single point symbol (see 5.3.7.5, 5.3.12, and figure 14).
     * Format: Graphic
     * Symbol Set: 10, 20, 25, 27, 40
     * Remarks: Determined by individual system implementations, not implemented by the renderer.
     * </pre>
     */
    public static final String S2_OFFSET_INDICATOR = "S2_OFFSET_INDICATOR";
    /**
     * <pre>
     * Unique Designation
     * An amplifier that uniquely identifies a particular symbol or track number. Identifies acquisitions number when used with SIGINT symbology.
     * Format: Alphanumeric - {1,30}
     * Symbol Set: All
     * Remarks:
     * </pre>
     */
    public static final String T_UNIQUE_DESIGNATION_1 = "T_UNIQUE_DESIGNATION_1";
    /**
     * <pre>
     * Lines - Unique Identifier or Primary Purpose
     * An amplifier that uniquely identifies a particular symbol or track number. Identifies acquisitions number when used with SIGINT symbology.
     * Format: Alphanumeric - Lines: {1,30}, Points: Alphanumeric - {1,7}
     * Symbol Set: 25
     * Remarks:
     * </pre>
     */
    public static final String T1_UNIQUE_DESIGNATION_2 = "T1_UNIQUE_DESIGNATION_2";
    /**
     * <pre>
     * Lines - Unique Identifier or Primary Purpose
     * An amplifier used to provide the designation of the Establishing Headquarters.
     * Format: Alphanumeric - {1,7}
     * Symbol Set: 25
     * Remarks: Used with Fire Support Control Measures only.
     * </pre>
     */
    public static final String T2_UNIQUE_DESIGNATION_3 = "T2_UNIQUE_DESIGNATION_3";
    /**
     * <pre>
     * Type
     * An amplifier for equipment that indicates types of equipment.
     * Format: Alphanumeric - {1,24}
     * Symbol Set: Not Installation(20), Activites(40), or CyberSpace(60)
     * Remarks:
     * Example:
     * AH-64 for Attack Helicopter
     * </pre>
     */
    public static final String V_EQUIP_TYPE = "V_EQUIP_TYPE";
    /**
     * <pre>
     * Date/Time Group (DTG) W
     * An amplifier for displaying a date-time group (DDHHMMSSZMONYYYY) or (DDHHMMZMMMYYYY) or "O/O" for On Order.
     * Field "W" is used to provide DTG or if used with field "W1" the start of a period of time.
     * Field "W1" is used to provide the end of a period of time.
     * Format: Alphanumeric - {3,16}
     * Symbol Set: 10, 15, 20, 25, 27, 40, 45
     * Remarks:
     * The date-time group is composed of a group of six numeric digits with a time zone suffix and the standardized three-letter abbreviation for the month followed by four digits representing the year.
     * The first pair of digits represents the day; the second pair, the hour; the third pair, the minutes. For automated systems, two digits may be added before the time zone suffix and after the minutes to designate seconds (see 5.3.7.8).
     * </pre>
     */
    public static final String W_DTG_1 = "W_DTG_1";
    /**
     * <pre>
     * Date/Time Group (DTG) W1
     * An amplifier for displaying a date-time group (DDHHMMSSZMONYYYY) or (DDHHMMZMMMYYYY) or "O/O" for On Order.
     * Field "W" is used to provide DTG or if used with field "W1" the start of a period of time.
     * Field "W1" is used to provide the end of a period of time.
     * Format: Alphanumeric - {3,16}
     * Symbol Set: 25
     * Remarks:
     * The date-time group is composed of a group of six numeric digits with a time zone suffix and the standardized three-letter abbreviation for the month followed by four digits representing the year.
     * The first pair of digits represents the day; the second pair, the hour; the third pair, the minutes. For automated systems, two digits may be added before the time zone suffix and after the minutes to designate seconds (see 5.3.7.8).
     * </pre>
     */
    public static final String W1_DTG_2 = "W1_DTG_2";
    /**
     * <pre>
     * Altitude/Depth
     * An amplifier that displays either altitude, flight level, depth for submerged objects or height of equipment or structures on the ground, the minimum, maximum and/or specific altitude (in feet or meters in relation to a reference datum) or depth (for submerged objects in feet below sea level). See 5.3.7.6 for content.
     * Format:
     * <br></br><img src="data:image/png;base64,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" alt="" />
     * Symbol Set: All but 40 and 60
     * Remarks:
     * SM = Statute Miles
     * DM = Data Miles
     * Notes:
     * The Renderer handles the whole value as a string for Single Point Icons.
     * For multipoints, it expects just a number and an accompanying attribute {@link MilStdAttributes#AltitudeUnits}
     * Default behavior assumes feet.
     * </pre>
     */
    public static final String X_ALTITUDE_DEPTH = "X_ALTITUDE_DEPTH";
    /**
     * <pre>
     * Location
     * An amplifier that displays a symbol's location in degrees, minutes and decimal minutes (or in MGRS, GARS, or other applicable display formats).
     * Format: Alphanumeric - {3,16}
     * Symbol Set: 10, 15, 20, 25, 27, 30, 40
     * Remarks: Exchange format is implementation specific.
     * </pre>
     */
    public static final String Y_LOCATION = "Y_LOCATION";
    /**
     * <pre>
     * Speed
     * An amplifier that displays velocity (see 5.3.7.7).
     * Format: Alphanumeric - {5,9} [#####] [XXX]
     * Symbol Set: NOT 20, 35, 40, 60
     * Remarks:
     * Two-part composite field.
     * Where # is the numeric value [1-999999999], and XXX is the unit of measure (KPH, KPS, MPH, NMH, KTS).
     * Note: There should be a space between the numeric and the unit of measure values.
     * </pre>
     */
    public static final String Z_SPEED = "Z_SPEED";
    /**
     * <pre>
     * Special C2 Headquarters
     * A amplifier that is contained inside the frame in place of the main icon and contains the name of the special C2 Headquarters.
     * Format: Alphanumeric - {1,9}
     * Symbol Set: 10
     * Remarks:
     * Examples: Named command such as SHAPE, PACOM, and joint, multinational, or coalition commands such as CJTF, JTF, or MJTF.
     * </pre>
     */
    public static final String AA_SPECIAL_C2_HQ = "AA_SPECIAL_C2_HQ";
    /**
     * <pre>
     * Feint/Dummy indicator
     * An amplifier that identifies an offensive or defensive unit, intended to draw the enemy's attention away from the area of the main attack.
     * Format: Graphic
     * Symbol Set: 10, 15, 20, 25
     * Remarks: Determined by SIDC position 8.
     * </pre>
     */
    public static final String AB_FEINT_DUMMY_INDICATOR = "AB_FEINT_DUMMY_INDICATOR";

    /**
     * <pre>
     * Platform Type
     * An amplifier that identifies the electronic identification for a pulsed or non-pulsed electromagnetic emission.
     * Format: Alphanumeric - {5,5}
     * Symbol Set: 10, 15
     * Remarks:
     * List of Values:
     * ELNOT = Electronic intelligence notation
     * CENOT = Communications intelligence notation
     * </pre>
     */
    public static final String AD_PLATFORM_TYPE = "AD_PLATFORM_TYPE";
    /**
     * <pre>
     * Platform Type
     * An amplifier that identifies equipment teardown time in minutes.
     * Format: Numeric – {1,3}
     * Symbol Set: 10(2525E), 15
     * Remarks:
     * </pre>
     */
    public static final String AE_EQUIPMENT_TEARDOWN_TIME = "AE_EQUIPMENT_TEARDOWN_TIME";
    /**
     * <pre>
     * Common Identifier
     * An amplifier to provide a common name used to identify an entity.
     * Format: Alphanumeric - {1,12}
     * Symbol Set: 10(2525E), 15, 27
     * Remarks: Example: "Hawk" for Hawk SAM system.
     * </pre>
     */
    public static final String AF_COMMON_IDENTIFIER = "AF_COMMON_IDENTIFIER";
    /**
     * <pre>
     * Auxiliary Equipment Indicator
     * An amplifier for equipment that indicates the presence of a towed sonar array (see 5.3.7.11, figure 14 and table IX).
     * Format: Graphic
     * Symbol Set: 15
     * Remarks: Determined by SIDC positions 9-10.
     * </pre>
     */
    public static final String AG_AUX_EQUIP_INDICATOR = "AG_AUX_EQUIP_INDICATOR";
    /**
     * <pre>
     * Area of Uncertainty
     * An amplifier that indicates the area where an object is most likely to be, based on the object's last report and
     * the reporting accuracy of the sensor that detected the object (see 5.3.7.13.1 and figure 18.
     * Format: Graphic (Alphanumeric for exchange only)
     * Symbol Set: 01, 05, 10, 15, 20, 25, 27, 30, 35, 40
     * Remarks:
     * The amplifier can be displayed as an ellipse, a bearing box, or a line of bearing, depending on the report received for the object.
     * Notes: Not implemented by the renderer
     * </pre>
     */
    public static final String AH_AREA_OF_UNCERTAINTY = "AH_AREA_OF_UNCERTAINTY";
    /**
     * <pre>
     * Dead Reckoning Trailer
     * An amplifier that identifies where an object should be located at present, given its last reported course and speed (see 5.3.7.13.2).
     * Format: Graphic
     * Symbol Set: 01, 05, 10, 15, 20, 25, 27, 30, 35, 40
     * Remarks:
     * Locally derived information. This datum is not exchanged.
     * Notes: Not implemented by the renderer
     * </pre>
     */
    public static final String AI_DEAD_RECKONING_TRAILER = "AI_DEAD_RECKONING_TRAILER";
    /**
     * <pre>
     * Speed Leader
     * An amplifier that depicts the speed and direction of movement of an object (see 5.3.7.13.3 and figure 18).
     * Format: Graphic
     * Symbol Set: 10, 15, 30, 35
     * Remarks: Set as "[Speed] [unit of measure] [angle in degrees if 3 characters "045", and in mils if 4 characters "0150]" like [100 KPH 045]
     * Valid units of measure are: KPH, KPS, MPH, NMH, KTS
     * Notes: It is recommended users handle this amplifier within their map engine many maps always show icons in the upright position when the map
     * is rotated, invalidating the line that's render with the symbol.
     * </pre>
     */
    public static final String AJ_SPEED_LEADER = "AJ_SPEED_LEADER";
    /**
     * <pre>
     * Pairing Line
     * An amplifier that connects two objects and is updated dynamically as the positions of the two objects change (see 5.3.7.13.4 and figure 18).
     * Format: Graphic
     * Symbol Set: NA
     * Remarks:
     * Notes: Not implemented by the renderer
     * </pre>
     */
    public static final String AK_PAIRING_LINE = "AK_PAIRING_LINE";
    /**
     * <pre>
     * Operational Condition
     * An amplifier that indicates operational condition or capacity.
     * Format: Graphic
     * Symbol Set: 01, 05,10, 15, 20, 30, 35
     * Remarks: Determined by SIDC position 7.
     * </pre>
     */
    public static final String AL_OPERATIONAL_CONDITION = "AL_OPERATIONAL_CONDITION";

    /**
     * <pre>
     * Distance
     * An amplifier that displays a minimum, maximum, or a specific distance (range, radius, width, length, etc.), in meters.
     * Format: Alphanumeric - {3,9} [#] [X]
     * Symbol Set: 25
     * Remarks:
     * "#" is the value (range 0-99999)
     * XXX is the distance unit.
     * There should be a space between the integer and the unit of measure values.
     * Where more than one distance is specified the AM amplifier will be suffixed with a numeric entry, e.g. AM1, AM2.
     * Note:
     * Renderer expects just a number or a comma delimited string of numbers and an accompanying attribute {@link MilStdAttributes#DistanceUnits}
     * Default behavior assumes meters.
     * </pre>
     */
    public static final String AM_DISTANCE = "AM_DISTANCE";
    /**
     * <pre>
     * Azimuth
     * An amplifier that displays an angle measured from true north to any other line in degrees.
     * Format: Alphanumeric - {7,8} [#####] [XXX]
     * Symbol Set: 25
     * Remarks:
     * For Degrees entries.
     * XXX DGT, where XXX is degrees from 000-359 and DGT is referenced to TRUE North.
     * For Mils entries.
     * XXXX MGT, where XXXX is MILS from 0000-6399 and MGT is referenced to TRUE North.
     * Where more than one angle is specified the AN amplifier will be suffixed with a numeric entry, e.g. AN1, AN2.
     * Note:
     * Renderer expects just a number or a comma delimited string of numbers
     * </pre>
     */
    public static final String AN_AZIMUTH = "AN_AZIMUTH";

    /**
     * Engagement Bar
     * A graphic amplifier placed immediately atop the symbol. May denote:
     * A) local/remote engagement status - 'R' for remote, 'B' for mix of local/remote, none for local
     * B) engagement status
     * C) weapon type.
     *
     * Format:
     * Composite list of values
     * Alphanumeric - {6,10}
     * A:BBB-CC
     * Symbol Set: 01, 05, 10, 30, 35
     * Remarks: See 5.3.7.15.3 for explanation of engagement bar structure and codes.
     * A - Type of Engagement
     * BBB - Engagement Stage
     * CC - Type of Weapon Assignment
     */
    public static final String AO_ENGAGEMENT_BAR = "AO_ENGAGEMENT_BAR";

    /**
     * <pre>
     * Target Number
     * An amplifier used in Fire Support operations to uniquely designate targets in accordance with STANAG 2934.
     * Format: Alphanumeric - {6,6} [XX][####]
     * Symbol Set: 25
     * Remarks:
     * Two-part composite field.
     * Where positions 1-2 are text, and positions 3-6 are numeric.
     * </pre>
     */
    public static final String AP_TARGET_NUMBER = "AP_TARGET_NUMBER";

    /**
     * <pre>
     * Target Number Extension
     * An amplifier used to identify a target number extension which is a sequentially assigned number identifying the individual elements in a target (MIL-STD-6017),
     * Format: Numeric – {2,3} [-##]
     * Symbol Set: 25
     * Remarks:
     * Position 1 is a dash (-) and positions 2-3 are numbers, from 1 through 15.
     * It is applicable only to the "Point or Single Target" symbol.
     * It is conditional upon the presence of the Target Number amplifier and is visually displayed appended to the Target Number amplifier.
     * </pre>
     */
    public static final String AP1_TARGET_NUMBER_EXTENSION = "AP1_TARGET_NUMBER_EXTENSION";

    /**
     * <pre>
     * Guarded Unit
     * An amplifier used during ballistic missile defense. Some tracks are designated as guarded by a particular unit.
     * Format: Alphanumeric - {6,6} [XX]
     * Symbol Set: 10(2525E), 15, 20(2525E), 30
     * Remarks:
     * Single value:
     * BG - Guarded Unit
     * </pre>
     */
    public static final String AQ_GUARDED_UNIT = "AQ_GUARDED_UNIT";

    /**
     * <pre>
     * Special Designator
     * An amplifier that identifies special track designators.
     * Format: Alphanumeric - {3,3}
     * Symbol Set: 10, 30, 35
     * Remarks:
     * List of values:
     * NRT - Non-Real Time.
     * SIG - Tactically Significant Tracks.
     * </pre>
     */
    public static final String AR_SPECIAL_DESIGNATOR = "AR_SPECIAL_DESIGNATOR";

    /**
     * <pre>
     * Country
     * A three-letter code that indicates the country of origin of the organization (US systems shall use GENC).
     * Format: Alphanumeric - {3,3}
     * Symbol Set: All
     * Remarks: Determined by SIDC positions 28-30.
     * </pre>
     */
    public static final String AS_COUNTRY = "AS_COUNTRY";

    /**
     * <pre>
     * Capacity of Installation
     * Capacity of installation displayed.
     * Format: Alphanumeric - {1,19} [#########] [XXXXXXXXXX]
     * Symbol Set: 2525E addition, not currently defined.  Probably 20
     * Remarks:
     * Two-part composite field.
     * Comprised of:
     * Quantity 0-99999999 followed by the unit of measure.
     * e.g. 400 Beds
     * </pre>
     */
    public static final String AT_CAPACITY_OF_INSTALLATION = "AT_CAPACITY_OF_INSTALLATION";

    /**
     * <pre>
     * Leadership
     * Identifies Leadership (ONLY IN DISMOUNTED INDIVIDUAL)
     * Format: Graphic
     * Symbol Set: 27(2525E)
     * Remarks: Determined by SIDC positions 9-10.
     * </pre>
     */
    public static final String AV_LEADERSHIP = "AV_LEADERSHIP";

    /**
     * <pre>
     * Headquarters Element
     * An amplifier that indicates what type of element of a headquarters is being represented, such as TOC, MAIN2.
     * Format: Alphanumeric {0,8}
     * Symbol Set: 10(2525E)
     * Remarks: Location currently undefined
     * </pre>
     */
    public static final String AW_HEADQUARTERS_ELEMENT = "AW_HEADQUARTERS_ELEMENT";

    /**
     * <pre>
     * Installation Composition
     * Indicates the component type of the installation
     * Format: Alphanumeric {3,7}
     * Symbol Set: 20(2525E)
     * Remarks: Location currently undefined
     * List of Values:
     * DEVELOP - Development.
     * RSRCH - Research.
     * PROD - Production.
     * SVC - Service.
     * STORE - Storage.
     * UTIL - Utility.
     * </pre>
     */
    public static final String AX_INSTALLATION_COMPOSITION = "AX_INSTALLATION_COMPOSITION";

    /**
     * <pre>
     * Network Identifier
     * Indicates the network the entity has privileges within.
     * Format: Alphanumeric
     * Symbol Set: undefined
     * Remarks:
     * The colour of the graphic shall be assignable by the operator.
     * Notes: undefined
     * </pre>
     */
    public static final String AY_NETWORK_IDENTIFIER = "AY_NETWORK_IDENTIFIER";

    //public static final String LENGTH = "Length";
    //public static final String WIDTH = "Width";
    //public static final String RADIUS = "Radius";
    //public static final String SEGMENT_DATA = "Segment Data";

    /**
     * Returns an Arraylist of the all the modifiers that appear as labels and are not
     * derived from the symbol code or are external to symbol data like offset indicator.
     * Also includes modifiers that control the shape of a symbol like AM &amp; AN.
     * @return {@link ArrayList}
     */
    public static ArrayList<String> GetModifierList()
    {
        ArrayList<String> list = new ArrayList<String>();

        //list.add(A_SYMBOL_ICON);//graphical, feeds off of symbol code,
        //list.add(B_ECHELON);//graphical, feeds off of symbol code,
        list.add(C_QUANTITY);
        list.add(D_TASK_FORCE_INDICATOR);//graphical, feeds off of symbol code,
        //list.add(E_FRAME_SHAPE_MODIFIER);//symbol frame, feeds off of symbol code, symbol set
        list.add(F_REINFORCED_REDUCED);//R = reinforced, D = reduced, RD = reinforced and reduced
        list.add(G_STAFF_COMMENTS);
        list.add(H_ADDITIONAL_INFO_1);
        list.add(H1_ADDITIONAL_INFO_2);
        list.add(J_EVALUATION_RATING);
        list.add(K_COMBAT_EFFECTIVENESS);
        list.add(L_SIGNATURE_EQUIP);
        list.add(M_HIGHER_FORMATION);
        list.add(N_HOSTILE);
        list.add(P_IFF_SIF_AIS);
        list.add(Q_DIRECTION_OF_MOVEMENT);//number in mils
        //list.add(R_MOBILITY_INDICATOR);//graphical, feeds off of symbol code,
        list.add(R2_SIGNIT_MOBILITY_INDICATOR);
        //list.add(S_HQ_STAFF_INDICATOR);//graphical, feeds off of symbol code,
        list.add(T_UNIQUE_DESIGNATION_1);
        list.add(T1_UNIQUE_DESIGNATION_2);
        list.add(V_EQUIP_TYPE);
        list.add(W_DTG_1);
        list.add(W1_DTG_2);
        list.add(X_ALTITUDE_DEPTH);
        list.add(Y_LOCATION);
        list.add(Z_SPEED);

        list.add(AA_SPECIAL_C2_HQ);
        //list.add(AB_FEINT_DUMMY_INDICATOR);//graphical, feeds off of symbol code,
        //list.add(AC_INSTALLATION);//graphical, feeds off of symbol code,
        list.add(AD_PLATFORM_TYPE);
        list.add(AE_EQUIPMENT_TEARDOWN_TIME);
        list.add(AF_COMMON_IDENTIFIER);
        //list.add(AG_AUX_EQUIP_INDICATOR);
        list.add(AH_AREA_OF_UNCERTAINTY);
        list.add(AI_DEAD_RECKONING_TRAILER);
        list.add(AJ_SPEED_LEADER);//graphical
        //list.add(AK_PAIRING_LINE);
        //list.add(AL_OPERATIONAL_CONDITION);//2525C ////graphical, feeds off of symbol code, SIDC positions 4
        list.add(AM_DISTANCE);
        list.add(AN_AZIMUTH);
        list.add(AO_ENGAGEMENT_BAR);//2525C
        list.add(AP_TARGET_NUMBER);
        list.add(AP1_TARGET_NUMBER_EXTENSION);
        list.add((AQ_GUARDED_UNIT));
        list.add((AR_SPECIAL_DESIGNATOR));
        //list.add((AS_COUNTRY));


        return list;
    }

    /**
     * Returns an ArrayList of the modifier constants that are determined by the symbol code
     * and therefore don't need their values to be manually typed or selected.
     * (Based on the comments in GetModifierList, GetUnitModifierList, and GetControlMeasureModifierList.)
     * @return {@link ArrayList}
     */
    public static ArrayList<String> GetSymbolCodeModifiersList() {
        ArrayList<String> list = new ArrayList<>();

        list.add(A_SYMBOL_ICON);//graphical, feeds off of symbol code,
        list.add(B_ECHELON);//graphical, feeds off of symbol code,
        list.add(D_TASK_FORCE_INDICATOR);//graphical, feeds off of symbol code,
        list.add(N_HOSTILE);//textual, feeds off symbol code (gets set to "ENY" if hostile)
        list.add(R_MOBILITY_INDICATOR);//graphical, feeds off of symbol code,
        list.add(S_HQ_STAFF_INDICATOR);//graphical, feeds off of symbol code,
        list.add(AB_FEINT_DUMMY_INDICATOR);//graphical, feeds off of symbol code,
        list.add(AG_AUX_EQUIP_INDICATOR);//graphical, feeds off of symbol code,
        list.add(AJ_SPEED_LEADER);//graphical
        list.add(AL_OPERATIONAL_CONDITION);//2525C ////graphical, feeds off of symbol code, SIDC positions 4
        list.add(AS_COUNTRY);//no need to manually enter code as the selector for the code is now implemented
        //list.add(-1); // TODO add missing modifiers (weather, others?) so this nameless default value doesn't get added

        return list;
    }

    /**
     * Returns an Arraylist of the modifier names for units
     * @return {@link ArrayList}
     */
    public static ArrayList<String> GetUnitModifierList()
    {
        ArrayList<String> list = new ArrayList<String>();

        //list.add(ModifierType.A_SYMBOL_ICON);//graphical, feeds off of symbol code, SIDC positions 3, 5-10
        //list.add(ModifierType.B_ECHELON);//graphical, feeds off of symbol code, SIDC positions 11-12
        list.add(C_QUANTITY);
        //list.add(D_TASK_FORCE_INDICATOR);//graphical, feeds off of symbol code, SIDC positions 11-12
        //list.add(E_FRAME_SHAPE_MODIFIER);//symbol frame, feeds off of symbol code, SIDC positions 3-4
        list.add(F_REINFORCED_REDUCED);//R = reinforced, D = reduced, RD = reinforced and reduced
        list.add(G_STAFF_COMMENTS);
        list.add(H_ADDITIONAL_INFO_1);
        //list.add(H1_ADDITIONAL_INFO_2);
        //list.add(H2_ADDITIONAL_INFO_3);
        list.add(J_EVALUATION_RATING);
        list.add(K_COMBAT_EFFECTIVENESS);
        list.add(L_SIGNATURE_EQUIP);
        list.add(M_HIGHER_FORMATION);
        list.add(N_HOSTILE);
        list.add(P_IFF_SIF_AIS);
        list.add(Q_DIRECTION_OF_MOVEMENT);//number in mils
        //list.add(R_MOBILITY_INDICATOR);//graphical, feeds off of symbol code, SIDC positions 11-12
        list.add(R2_SIGNIT_MOBILITY_INDICATOR);
        //list.add(S_HQ_STAFF_OR_OFFSET_INDICATOR);//graphical, feeds off of symbol code, SIDC positions 11-12
        list.add(T_UNIQUE_DESIGNATION_1);
        //list.add(T1_UNIQUE_DESIGNATION_2);
        list.add(V_EQUIP_TYPE);
        list.add(W_DTG_1);
        list.add(W1_DTG_2);
        list.add(X_ALTITUDE_DEPTH);
        list.add(Y_LOCATION);
        list.add(Z_SPEED);

        list.add(AA_SPECIAL_C2_HQ);
        //list.add(AB_FEINT_DUMMY_INDICATOR);//graphical, feeds off of symbol code, SIDC positions 11-12
        //list.add(AC_INSTALLATION);//graphical, feeds off of symbol code, SIDC positions 11-12
        list.add(AD_PLATFORM_TYPE);
        list.add(AE_EQUIPMENT_TEARDOWN_TIME);
        list.add(AF_COMMON_IDENTIFIER);
        //list.add(AG_AUX_EQUIP_INDICATOR);//graphical
        //list.add(AH_AREA_OF_UNCERTAINTY);//graphical and on client of renderer
        //list.add(AI_DEAD_RECKONING_TRAILER);//graphical
        list.add(AJ_SPEED_LEADER);
        list.add(AK_PAIRING_LINE);
        //list.add(AL_OPERATIONAL_CONDITION);//2525C ////graphical, feeds off of symbol code, SIDC positions 4
        list.add(AO_ENGAGEMENT_BAR);//2525C
        list.add(AP_TARGET_NUMBER);
        list.add(AP1_TARGET_NUMBER_EXTENSION);
        list.add((AQ_GUARDED_UNIT));
        list.add((AR_SPECIAL_DESIGNATOR));
        list.add((AS_COUNTRY));


        return list;
    }

    /**
     * Returns an Arraylist of the modifiers for Control Measures
     * @return {@link ArrayList}
     */
    public static ArrayList<String> GetControlMeasureModifierList()
    {
        ArrayList<String> list = new ArrayList<String>();

        //list.add(ModifierType.B_ECHELON);//graphical, feeds off of symbol code,
        list.add(C_QUANTITY);
        list.add(H_ADDITIONAL_INFO_1);
        list.add(H1_ADDITIONAL_INFO_2);
        list.add(N_HOSTILE);
        list.add(Q_DIRECTION_OF_MOVEMENT);//number in mils
        //list.add(S2);//graphical, up to renderer client
        list.add(T_UNIQUE_DESIGNATION_1);
        list.add(T1_UNIQUE_DESIGNATION_2);
        list.add(V_EQUIP_TYPE);
        list.add(W_DTG_1);
        list.add(W1_DTG_2);
        list.add(X_ALTITUDE_DEPTH);
        list.add(Y_LOCATION);

        list.add(AM_DISTANCE);
        list.add(AN_AZIMUTH);
        list.add(AP_TARGET_NUMBER);
        list.add(AP1_TARGET_NUMBER_EXTENSION);
        list.add((AS_COUNTRY));




        return list;
    }

    /**
     *
     * @param modifier like Modifiers.C_QUANTITY
     * @return modifier name based on modifier constants
     */
    public static String getModifierName(String modifier)
    {
        switch(modifier)
        {
            case A_SYMBOL_ICON:
                return "Symbol Icon";
            case B_ECHELON:
                return "Echelon";
            case C_QUANTITY:
                return "Quantity";
            case D_TASK_FORCE_INDICATOR:
                return "Task Force Indicator";
            case F_REINFORCED_REDUCED:
                return "Reinforce Reduced";
            case G_STAFF_COMMENTS:
                return "Staff Comments";
            case H_ADDITIONAL_INFO_1:
                return "Additional Info 1";
            case H1_ADDITIONAL_INFO_2:
                return "Additional Info 2";
            case J_EVALUATION_RATING:
                return "Evaluation Rating";
            case K_COMBAT_EFFECTIVENESS:
                return "Combat Effectiveness";
            case L_SIGNATURE_EQUIP:
                return "Signature Equipment";
            case M_HIGHER_FORMATION:
                return "Higher Formation";
            case N_HOSTILE:
                return "Hostile";
            case P_IFF_SIF_AIS:
                return "IFF SIF AIS";
            case Q_DIRECTION_OF_MOVEMENT:
                return "Direction of Movement";
            case R_MOBILITY_INDICATOR:
                return "Mobility Indicator";
            case R2_SIGNIT_MOBILITY_INDICATOR:
                return "Signals Intelligence Mobility Indicator";
            case S_HQ_STAFF_INDICATOR:
                return "HQ Staff";
            case S2_OFFSET_INDICATOR:
                return "Offset Indicator";
            case T_UNIQUE_DESIGNATION_1:
                return "Unique Designation 1";
            case T1_UNIQUE_DESIGNATION_2:
                return "Unique Designation 2";
            case V_EQUIP_TYPE:
                return "Equipment Type";
            case W_DTG_1:
                return "Date Time Group 1";
            case W1_DTG_2:
                return "Date Time Group 2";
            case X_ALTITUDE_DEPTH:
                return "Altitude Depth";
            case Y_LOCATION:
                return "Location";
            case Z_SPEED:
                return "Speed";
            case AA_SPECIAL_C2_HQ:
                return "Special C2 HQ";
            case AB_FEINT_DUMMY_INDICATOR:
                return "Feint Dummy Indicator";
            case AD_PLATFORM_TYPE:
                return "Platform Type";
            case AE_EQUIPMENT_TEARDOWN_TIME:
                return "Equipment Teardown Time";
            case AF_COMMON_IDENTIFIER:
                return "Common Identifier";
            case AG_AUX_EQUIP_INDICATOR:
                return "Auxiliary Equipment Indicator";
            case AH_AREA_OF_UNCERTAINTY:
                return "Area of Uncertainty";
            case AI_DEAD_RECKONING_TRAILER:
                return "Dead Reckoning Trailer";
            case AJ_SPEED_LEADER:
                return "Speed Leader";
            case AK_PAIRING_LINE:
                return "Pairing Line";
            case AL_OPERATIONAL_CONDITION:
                return "Operational Condition";
            case AM_DISTANCE:
                return "Distance";
            case AN_AZIMUTH:
                return "Azimuth";
            case AO_ENGAGEMENT_BAR:
                return "Engagement Bar";//*/
            case AP_TARGET_NUMBER:
                return "Target Number";
            case AP1_TARGET_NUMBER_EXTENSION:
                return "Target Number Extension";
            case AQ_GUARDED_UNIT:
                return "Guarded Unit";
            case AR_SPECIAL_DESIGNATOR:
                return "Special Designator";
            case AS_COUNTRY:
                return "Country";
            default:
                return "";

        }
    }

    /**
     *
     * @param modifier like Modifiers.C_QUANTITY
     * @return modifier name based on modifier constants
     */
    public static String getModifierLetterCode(String modifier) {
        switch (modifier) {
            case A_SYMBOL_ICON:
                return "A";
            case B_ECHELON:
                return "B";
            case C_QUANTITY:
                return "C";
            case D_TASK_FORCE_INDICATOR:
                return "D";
            case F_REINFORCED_REDUCED:
                return "F";
            case G_STAFF_COMMENTS:
                return "G";
            case H_ADDITIONAL_INFO_1:
                return "H";
            case H1_ADDITIONAL_INFO_2:
                return "H1";
            case J_EVALUATION_RATING:
                return "J";
            case K_COMBAT_EFFECTIVENESS:
                return "K";
            case L_SIGNATURE_EQUIP:
                return "L";
            case M_HIGHER_FORMATION:
                return "M";
            case N_HOSTILE:
                return "N";
            case P_IFF_SIF_AIS:
                return "P";
            case Q_DIRECTION_OF_MOVEMENT:
                return "Q";
            case R_MOBILITY_INDICATOR:
                return "R";
            case R2_SIGNIT_MOBILITY_INDICATOR:
                return "R2";
            case S_HQ_STAFF_INDICATOR:
                return "S";
            case S2_OFFSET_INDICATOR:
                return "S2";
            case T_UNIQUE_DESIGNATION_1:
                return "T";
            case T1_UNIQUE_DESIGNATION_2:
                return "T1";
            case V_EQUIP_TYPE:
                return "V";
            case W_DTG_1:
                return "W";
            case W1_DTG_2:
                return "W1";
            case X_ALTITUDE_DEPTH:
                return "X";
            case Y_LOCATION:
                return "Y";
            case Z_SPEED:
                return "Z";
            case AA_SPECIAL_C2_HQ:
                return "AA";
            case AB_FEINT_DUMMY_INDICATOR:
                return "AB";
            case AD_PLATFORM_TYPE:
                return "AD";
            case AE_EQUIPMENT_TEARDOWN_TIME:
                return "AE";
            case AF_COMMON_IDENTIFIER:
                return "AF";
            case AG_AUX_EQUIP_INDICATOR:
                return "AG";
            case AH_AREA_OF_UNCERTAINTY:
                return "AH";
            case AI_DEAD_RECKONING_TRAILER:
                return "AI";
            case AJ_SPEED_LEADER:
                return "AJ";
            case AK_PAIRING_LINE:
                return "AK";
            case AL_OPERATIONAL_CONDITION:
                return "AL";
            case AM_DISTANCE:
                return "AM";
            case AN_AZIMUTH:
                return "AN";
            case AO_ENGAGEMENT_BAR:
                return "AO";
            case AP_TARGET_NUMBER:
                return "AP";
            case AP1_TARGET_NUMBER_EXTENSION:
                return "AP1";
            case AQ_GUARDED_UNIT:
                return "AQ";
            case AR_SPECIAL_DESIGNATOR:
                return "AR";
            case AS_COUNTRY:
                return "AS";
            default:
                return "";

        }
    }

    /**
     * Returns the description for a modifier constant
     * @param modifier {@link Modifiers}
     * @return {@link String}
     */
    public static String getModifierDescription(String modifier) {
        switch (modifier) {
            case A_SYMBOL_ICON:
                return "The innermost part of a symbol, comprised of an icon and optional modifiers, that represents a joint military object (see 5.3.4).\n" +
                        "Determined by SIDC positions 11-20.";
            case B_ECHELON:
                return "A graphic amplifier in a unit symbol that identifies command level (see 5.3.7.1 and table D-III).";
            case C_QUANTITY:
                return "A text amplifier in an equipment symbol that identifies the number of items present.\n" +
                        "Examples include: \"350 Beds\" or \"50 Gallons\"";
            case D_TASK_FORCE_INDICATOR:
                return "A graphic amplifier that identifies a unit or activities symbol as a task force (see 5.3.7.2 and figure 13).\n" +
                        "Determined by SIDC position 8.";
            case F_REINFORCED_REDUCED:
                return "A text modifier in a unit symbol that displays (+) for reinforced, (-) for reduced,(+) reinforced and reduced.\n" +
                        "R = reinforced,D = reduced, RD = reinforced and reduced";
            case G_STAFF_COMMENTS:
            case H_ADDITIONAL_INFO_1:
                return "A text modifier for units, equipment and installations; content is implementation specific.";
            case H1_ADDITIONAL_INFO_2:
                return "An alphanumeric text amplifier used to provide an up to a three-character letter field acronym to describe a point that is not already listed.";
            case J_EVALUATION_RATING:
                return "A text modifier for units, equipment, and installations that consists of\n "
                        + "a one letter reliability rating and a one-number credibility rating.\n"
                        + "Reliability Ratings: A-completely reliable, B-usually reliable,\n "
                        + "C-fairly reliable, D-not usually reliable, E-unreliable,\n "
                        + "F-reliability cannot be judged.\n"
                        + "Credibility Ratings: 1-confirmed by other sources,\n"
                        + "2-probably true, 3-possibly true, 4-doubtfully true,\n"
                        + "5-improbable, 6-truth cannot be judged.\n"
                        + "Example \"A1\"";
            case K_COMBAT_EFFECTIVENESS:
                return "A text modifier for units and installations that indicates unit effectiveness or installation capability." +
                        "FO - Fully Operational\n" +
                        "SO - Substantially operational\n" +
                        "MO - Marginally operational\n" +
                        "NO - Not operational\n" +
                        "UNK - Unknown";
            case L_SIGNATURE_EQUIP:
                return "A text modifier for hostile equipment; \"!\" indicates detectable electronic signatures.";
            case M_HIGHER_FORMATION:
                return "An amplifier for units that indicates number or title of higher echelon command (corps are designated by Roman numerals).";
            case N_HOSTILE:
                return "An amplifier for equipment; letters \"ENY\" denote hostile symbols." +
                        "Determined by SIDC position 4.";
            case P_IFF_SIF_AIS:
                return "An amplifier displaying IFF/SIF/AIS Identification modes and codes." +
                        "Alphanumeric - {4,4}\n" +
                        "MODE 1:##, where ## is a two-digit octal number (0-7). The rightmost digit is limited to 0-3.\n" +
                        "[1:##]\n" +
                        "Note: ‘MODE' is not required when displaying.\n" +
                        "Alphanumeric - {6,6}\n" +
                        "MODE 2:####, where #### is a four-digit octal number (0-7).\n" +
                        "[2:####]\n" +
                        "Alphanumeric - {6,6}\n" +
                        "MODE 3:####, where #### is a four-digit octal number (0-7).\n" +
                        "[3:####]\n" +
                        "Alphanumeric - {6,6}\n" +
                        "MODE 5:####, where #### is a 4-digit hexadecimal number (0-F). The leftmost digit is limited from 0 to 3.\n" +
                        "[5:####]\n" +
                        "Alphanumeric - {8,8}\n" +
                        "MODE S:XXXXXX\n" +
                        "[S:XXXXXX]\n" +
                        "Alphanumeric - {x,x}\n" +
                        "AIS IMO:XXXXXX";
            case Q_DIRECTION_OF_MOVEMENT:
                return "An amplifier consisting of a line with arrow that identifies the direction of movement or intended movement of an object (see 5.3.7.9 and figure 14).\n";
            case R_MOBILITY_INDICATOR:
                return "An amplifier that depicts the mobility of an object not intrinsic to the entity itself (see 5.3.7.9, figure 14 and table IX).\n" +
                        "Determined by SIDC positions 9-10";
            case R2_SIGNIT_MOBILITY_INDICATOR:
                return "An amplifier that indicates the mobility of a SIGINT unit." +
                        "List of Values:\n" +
                        "M = Mobile,\n" +
                        "S = Static\n" +
                        "U = Uncertain.";
            case S_HQ_STAFF_INDICATOR:
                return "An amplifier for units, installations and activities that identifies them as a headquarters (see figure 14 and figure 13).\n" +
                        "Determined by SIDC position 8.";
            case S2_OFFSET_INDICATOR:
                return "An amplifier used to indicate the offset or precise location of a single point symbol (see 5.3.7.5, 5.3.12, and figure 14).";
            case T_UNIQUE_DESIGNATION_1:
                return "An amplifier that uniquely identifies a particular symbol or track number. Identifies acquisitions number when used with SIGINT symbology.";
            case T1_UNIQUE_DESIGNATION_2:
                return "Lines - Second unique identifier for boundaries.\n" +
                        "Lines - Other than phase lines that have a specific purpose, (for example, restrictive fire line RFL),\n " +
                        "   should have the primary purpose labelled on top of the line at both ends of the line inside the\n " +
                        "   lateral boundaries or as often as necessary for clarity.\n" +
                        "Points - Provides the unit servicing the point";
            case V_EQUIP_TYPE:
                return "An amplifier for equipment that indicates types of equipment.\n" +
                        "Example: " +
                        "AH-64 for Attack Helicopter";
            case W_DTG_1:
            case W1_DTG_2:
                return "An amplifier for displaying a date-time group (DDHHMMSSZMONYYYY) or (DDHHMMZMMMYYYY) or \"O/O\" for On Order.\n" +
                        "Field \"W\" is used to provide DTG or if used with field \"W1\" the start of a period of time.\n" +
                        "Field \"W1\" is used to provide the end of a period of time.\n" +
                        "The date-time group is composed of a group of six numeric digits with a time zone suffix and the standardized three-letter abbreviation for the month followed by four digits representing the year.\n" +
                        "The first pair of digits represents the day; the second pair, the hour; the third pair, the minutes. For automated systems, two digits may be added before the time zone suffix and after the minutes to designate seconds (see 5.3.7.8).";
            case X_ALTITUDE_DEPTH:
                return "An amplifier that displays either altitude, flight level, depth for submerged objects or height of equipment or structures on the ground, the minimum, maximum and/or specific altitude (in feet or meters in relation to a reference datum) or depth (for submerged objects in feet below sea level). See 5.3.7.6 for content.";
            case Y_LOCATION:
                return "An amplifier that displays a symbol's location in degrees, minutes and decimal minutes (or in MGRS, GARS, or other applicable display formats).";
            case Z_SPEED:
                return "An amplifier that displays velocity (see 5.3.7.7).";
            case AA_SPECIAL_C2_HQ:
                return "A amplifier that is contained inside the frame in place of the main icon and contains the name of the special C2 Headquarters.";
            case AB_FEINT_DUMMY_INDICATOR:
                return "An amplifier that identifies an offensive or defensive unit, intended to draw the enemy's attention away from the area of the main attack.\n" +
                        "Determined by SIDC position 8.";
            case AD_PLATFORM_TYPE:
                return "Electronic intelligence notation (ELNOT) or communications intelligence notation (CENOT)";
            case AE_EQUIPMENT_TEARDOWN_TIME:
                return "Equipment teardown time in minutes.";
            case AF_COMMON_IDENTIFIER:
                return "Example: \"Hawk\" for Hawk SAM system.";
            case AG_AUX_EQUIP_INDICATOR:
                return "Towed sonar array indicator: A graphic modifier for equipment that indicates the presence of a towed sonar array (see 5.3.7.10, figure 13 and table IX).\n" +
                        "Determined by SIDC positions 9-10";
            case AH_AREA_OF_UNCERTAINTY:
                return "A graphic modifier for units, equipment and installations that indicates the area where an object is most likely to be, based on the object's last report and the reporting accuracy of the sensor that detected the object (see 5.3.7.12.1 and table D-III).";
            case AI_DEAD_RECKONING_TRAILER:
                return "A graphic amplifier for units and equipment that identifies where an object should be located at present, given its last reported course and speed (see 5.3.7.12.2).";
            case AJ_SPEED_LEADER:
                return "A graphic amplifier for units, equipment and installations that depicts the speed and direction of movement of an object (see 5.3.7.12.3 and figure 17).";
            case AK_PAIRING_LINE:
                return "A graphic amplifier for units, equipment and installations that connects two objects and is updated dynamically as the positions of the two objects change (see 5.3.7.12.4 and figure 17).";
            case AL_OPERATIONAL_CONDITION:
                return "A graphic amplifier for equipment or installations that indicates operational condition or capacity.";
            case AM_DISTANCE:
                return "An amplifier that displays a minimum, maximum, or a specific distance (range, radius, width, length, etc.), in meters.";
            case AN_AZIMUTH:
                return "An amplifier that displays an angle measured from true north to any other line in degrees.";
            case AO_ENGAGEMENT_BAR:
                return "An amplifier placed immediately atop the symbol.\n" +
                        "May denote: 1) local/remote status 2) engagement status 3) weapon type.";
            case AP_TARGET_NUMBER:
                return "An amplifier used in Fire Support operations to uniquely designate targets in accordance with STANAG 2934.";
            case AP1_TARGET_NUMBER_EXTENSION:
                return "An amplifier used to identify a target number extension which is a sequentially assigned number identifying the individual elements in a target (MIL-STD-6017),";
            case AQ_GUARDED_UNIT:
                return "An amplifier used during ballistic missile defense. Some tracks are designated as guarded by a particular unit.";
            case AR_SPECIAL_DESIGNATOR:
                return "An amplifier that identifies special track designators.";
            case AS_COUNTRY:
                return "A three-letter code that indicates the country of origin of the organization (US systems shall use GENC).";
            default:
                return "";

        }
    }

    /**
     * go from "T" or "T1" to integer constant values 9 and 10
     * @param modLetter {@link String}
     * @return {@link Modifiers}
     */
    public static String getModifierKey(String modLetter) {
        switch (modLetter) {
            case "A":
                return A_SYMBOL_ICON;
            case "B":
                return B_ECHELON;
            case "C":
                return C_QUANTITY;
            case "D":
                return D_TASK_FORCE_INDICATOR;
            case "F":
                return F_REINFORCED_REDUCED;
            case "G":
                return G_STAFF_COMMENTS;
            case "H":
                return H_ADDITIONAL_INFO_1;
            case "H1":
                return H1_ADDITIONAL_INFO_2;
            case "J":
                return J_EVALUATION_RATING;
            case "K":
                return K_COMBAT_EFFECTIVENESS;
            case "L":
                return L_SIGNATURE_EQUIP;
            case "M":
                return M_HIGHER_FORMATION;
            case "N":
                return N_HOSTILE;
            case "P":
                return P_IFF_SIF_AIS;
            case "Q":
                return Q_DIRECTION_OF_MOVEMENT;
            case "R":
                return R_MOBILITY_INDICATOR;
            case "R2":
                return R2_SIGNIT_MOBILITY_INDICATOR;
            case "S":
                return S_HQ_STAFF_INDICATOR;
            case "S2":
                return S2_OFFSET_INDICATOR;
            case "T":
                return T_UNIQUE_DESIGNATION_1;
            case "T1":
                return T1_UNIQUE_DESIGNATION_2;
            case "V":
                return V_EQUIP_TYPE;
            case "W":
                return W_DTG_1;
            case "W1":
                return W1_DTG_2;
            case "X":
                return X_ALTITUDE_DEPTH;
            case "Y":
                return Y_LOCATION;
            case "Z":
                return Z_SPEED;
            case "AA":
                return AA_SPECIAL_C2_HQ;
            case "AB":
                return AB_FEINT_DUMMY_INDICATOR;
            case "AD":
                return AD_PLATFORM_TYPE;
            case "AE":
                return AE_EQUIPMENT_TEARDOWN_TIME;
            case "AF":
                return AF_COMMON_IDENTIFIER;
            case "AG":
                return AG_AUX_EQUIP_INDICATOR;
            case "AH":
                return AH_AREA_OF_UNCERTAINTY;
            case "AI":
                return AI_DEAD_RECKONING_TRAILER;
            case "AJ":
                return AJ_SPEED_LEADER;
            case "AK":
                return AK_PAIRING_LINE;
            case "AL":
                return AL_OPERATIONAL_CONDITION;
            case "AM":
                return AM_DISTANCE;
            case "AN":
                return AN_AZIMUTH;
            case "AO":
                return AO_ENGAGEMENT_BAR;
            case "AP":
                return AP_TARGET_NUMBER;
            case "AP1":
                return AP1_TARGET_NUMBER_EXTENSION;
            case "AQ":
                return AQ_GUARDED_UNIT;
            case "AR":
                return AR_SPECIAL_DESIGNATOR;
            case "AS":
                return AS_COUNTRY;
            default:
                return null;

        }
    }
}

