// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.refcodes.data.AnsiEscapeCode;
import org.refcodes.data.AsciiColorPalette;
import org.refcodes.data.CommandArgPrefix;
import org.refcodes.data.ConsoleDimension;
import org.refcodes.data.Delimiter;
import org.refcodes.data.License;
import org.refcodes.data.Literal;
import org.refcodes.graphical.BoxBorderMode;
import org.refcodes.runtime.RuntimeUtility;
import org.refcodes.runtime.Terminal;
import org.refcodes.struct.PropertyImpl;
import org.refcodes.textual.AsciiArtBuilder;
import org.refcodes.textual.AsciiArtMode;
import org.refcodes.textual.ColumnWidthType;
import org.refcodes.textual.Font;
import org.refcodes.textual.FontFamily;
import org.refcodes.textual.FontStyle;
import org.refcodes.textual.HorizAlignTextBuilder;
import org.refcodes.textual.HorizAlignTextMode;
import org.refcodes.textual.PixmapRatioMode;
import org.refcodes.textual.SplitTextMode;
import org.refcodes.textual.TableBuilder;
import org.refcodes.textual.TableStyle;
import org.refcodes.textual.TextBlockBuilder;
import org.refcodes.textual.TextBorderBuilder;
import org.refcodes.textual.TextBoxGrid;
import org.refcodes.textual.TextBoxStyle;
import org.refcodes.textual.TextLineBuilder;
import org.refcodes.textual.VerboseTextBuilder;

/**
 * A straightforward implementation of the {@link ArgsParser} interface. The
 * constructor only provides means to set the required attributes as the
 * attributes to be adjusted optionally are already sufficiently pre-configured.
 * For adjusting them, a flavor of the Builder-Pattern is provided with which
 * you can easily chain the configuration of this instance; as them methods
 * return the instance of this class being configured. This helps to prevent the
 * telescoping constructor anti-pattern.
 * <p>
 * The {@link SyntaxNotation} is pre-set with the
 * {@link SyntaxNotation#REFCODES} notation.
 * <p>
 * The console width id pre-configured with the console's width as determined by
 * the <code>SystemUtility.getTerminalWidth()</code>.
 * <p>
 * The standard out {@link PrintStream} is pre-configured with the
 * {@link System#out} {@link PrintStream}.
 * <p>
 * The newline characters to be used for line breaks is "\r\n" on Windows
 * machines and "\"n" on all other machines as of the
 * <code>SystemUtility.getLineBreak()</code>.
 *
 * @see "http://en.wikipedia.org/wiki/Builder_pattern"
 */
public class ArgsParserImpl implements ArgsParser {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private List<Example> _examples = null;
	private Font _bannerFont = new Font( FontFamily.DIALOG, FontStyle.PLAIN, 12 );
	private char[] _bannerFontPalette = AsciiColorPalette.HALFTONE_GRAY.getPalette();
	private ArgsSyntax _rootArgsSyntax;
	private SyntaxNotation _syntaxNotation = SyntaxNotation.REFCODES;
	private int _consoleWidth = Terminal.toPreferredTerminalWidth();
	private int _maxConsoleWidth = -1;
	private String _lineBreak = Terminal.getLineBreak();
	private String _title = null;
	private String _name = "foobar";
	private String _description = "See the syntax declaration for usage, see the descriptions for the short- and the long-options. Option arguments are noted in angle brackets.";
	private String _licenseNote = License.LICENSE_NOTE.getText();
	private String _copyrightNote = License.COPYRIGHT_NOTE.getText();;
	private boolean _isEscCodesEnabled = Terminal.isAnsiTerminalPreferred();
	private String _resetEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.RESET );
	private String _commandEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.ITALIC );
	private String _paramEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.FAINT, AnsiEscapeCode.ITALIC );
	private String _optEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.BOLD );
	private String _descriptionEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR );
	private String _bannerEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.BOLD );
	private String _bannerBorderEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.FAINT );
	private String _lineSeparatorEscCode = AnsiEscapeCode.toEscapeSequence( AnsiEscapeCode.DEFAULT_FOREGROUND_COLOR, AnsiEscapeCode.FAINT );
	private TextBoxGrid _textBoxGrid = TextBoxStyle.toRuntimeTextBoxStyle().getTextBoxGrid();
	private char _separatorLnChar = _textBoxGrid.getInnerLine();
	private boolean _hasOverrideSeparatorLnChar = false;
	protected PrintStream _stdStream = RuntimeUtility.toSystemOut();
	protected PrintStream _errStream = RuntimeUtility.toSystemErr();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the {@link ArgsParser} instance without any restrictions to
	 * the parsed arguments. The constructor only provides means to set the
	 * required attributes as the attributes to be adjusted optionally are
	 * already sufficiently pre-configured. For adjusting them, a flavor of the
	 * Builder-Pattern is provided with which you can easily chain the
	 * configuration of this instance; as them methods return the instance of
	 * this class being configured.
	 */
	public ArgsParserImpl() {
		_rootArgsSyntax = null;
	}

	/**
	 * Constructs the {@link ArgsParser} instance with the given root
	 * {@link ArgsSyntax} and the default {@link SyntaxNotation#REFCODES}. The
	 * constructor only provides means to set the required attributes as the
	 * attributes to be adjusted optionally are already sufficiently
	 * pre-configured. For adjusting them, a flavor of the Builder-Pattern is
	 * provided with which you can easily chain the configuration of this
	 * instance; as them methods return the instance of this class being
	 * configured.
	 * 
	 * @param aRootArgsSyntax The root condition being the node from which
	 *        parsing the command line arguments starts.
	 */
	public ArgsParserImpl( ArgsSyntax aRootArgsSyntax ) {
		_rootArgsSyntax = aRootArgsSyntax;
	}

	/**
	 * Constructs the {@link ArgsParser} instance with the given root
	 * {@link Option} and the default {@link SyntaxNotation#REFCODES}. The
	 * constructor only provides means to set the required attributes as the
	 * attributes to be adjusted optionally are already sufficiently
	 * pre-configured. For adjusting them, a flavor of the Builder-Pattern is
	 * provided with which you can easily chain the configuration of this
	 * instance; as them methods return the instance of this class being
	 * configured.
	 * 
	 * @param aRootOption The root option being the node from which parsing the
	 *        command line arguments starts.
	 */
	public ArgsParserImpl( Option<?> aRootOption ) {
		_rootArgsSyntax = new OptionCondition( aRootOption );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLineSeparatorEscapeCode() {
		return _lineSeparatorEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLineSeparatorEscapeCode( String aLineSeparatorEscCode ) {
		_lineSeparatorEscCode = aLineSeparatorEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBannerEscapeCode() {
		return _bannerEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBannerBorderEscapeCode() {
		return _bannerBorderEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerEscapeCode( String aBannerEscCode ) {
		_bannerEscCode = aBannerEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerBorderEscapeCode( String aBannerBorderEscCode ) {
		_bannerBorderEscCode = aBannerBorderEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getParamEscapeCode() {
		return _paramEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setParamEscapeCode( String aParamEscCode ) {
		_paramEscCode = aParamEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getOptionEscapeCode() {
		return _optEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptionEscapeCode( String aOptEscCode ) {
		_optEscCode = aOptEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void errorLn( String aLine ) {
		if ( aLine == null ) aLine = "null";
		String[] theLines = new TextBlockBuilder().withText( aLine ).withColumnWidth( _consoleWidth ).withSplitTextMode( SplitTextMode.AT_SPACE ).toStrings();
		_errStream.print( fromTextBlock( theLines, toLineBreak() ) );
		_errStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<? extends Operand<?>> evalArgs( String[] aArgs ) throws ArgsSyntaxException {
		if ( _rootArgsSyntax == null ) {
			return fromArgs( aArgs, getDelimiter() );
		}
		else {
			List<? extends Operand<?>> theOperands = _rootArgsSyntax.parseArgs( aArgs );
			String[] theSuperflousArgs = CliUtility.toDiff( aArgs, theOperands );
			if ( theSuperflousArgs != null && theSuperflousArgs.length > 0 ) {
				throw new SuperfluousArgsException( theSuperflousArgs, "Superfluous command arguments " + new VerboseTextBuilder().withElements( theSuperflousArgs ).toString() + " were provided but cannot be evaluated or are not supported as of the given combination of arguments." );
			}
			return theOperands;
		}

	}

	/**
	 * From text block.
	 *
	 * @param aTextBlock the text block
	 * @param aDelimeter the deimeter
	 * 
	 * @return the string
	 */
	private String fromTextBlock( String[] aTextBlock, String aDelimeter ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( String eString : aTextBlock ) {
			if ( aDelimeter != null && aDelimeter.length() != 0 ) {
				if ( theBuilder.length() > 0 ) {
					theBuilder.append( aDelimeter );
				}
			}
			theBuilder.append( eString );
		}
		return theBuilder.toString() + toLineBreak();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getCopyrightNote() {
		return _copyrightNote;
	}

	/**
	 * Returns the delimiter to be used by colliding command line args when
	 * creating non colliding arg's aliases (keys for key/value-pairs).
	 * 
	 * @return The according delimiter.
	 */
	protected char getDelimiter() {
		return Delimiter.INDEX.getChar();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return _description;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLicenseNote() {
		return _licenseNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return _name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ArgsSyntax getArgsSyntax() {
		return _rootArgsSyntax;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResetEscapeCode() {
		return _resetEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setResetEscapeCode( String aResetEscCode ) {
		_resetEscCode = aResetEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTitle() {
		return _title;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printBanner() {
		int theBannerWidth = _consoleWidth - 4;
		String[] theCanvas = new AsciiArtBuilder().withText( _title != null ? _title : _name ).withFont( _bannerFont ).withAsciiColors( _bannerFontPalette ).withColumnWidth( theBannerWidth ).withAsciiArtMode( AsciiArtMode.NORMAL ).withPixmapRatioMode( PixmapRatioMode.ONE_HALF ).toStrings();
		boolean hasBorder = hasBorder( theCanvas );
		if ( hasBorder ) {
			theBannerWidth = _consoleWidth - 2;
		}
		theCanvas = new AsciiArtBuilder().withText( _title != null ? _title : _name ).withFont( _bannerFont ).withAsciiColors( _bannerFontPalette ).withColumnWidth( theBannerWidth ).withAsciiArtMode( AsciiArtMode.NORMAL ).withPixmapRatioMode( PixmapRatioMode.ONE_HALF ).toStrings();
		theCanvas = new HorizAlignTextBuilder().withHorizAlignTextMode( HorizAlignTextMode.CENTER ).withText( theCanvas ).withColumnWidth( theBannerWidth ).withFillChar( ' ' ).toStrings();
		if ( !hasBorder ) {
			theCanvas = new TextBorderBuilder().withBoxBorderMode( BoxBorderMode.ALL ).withText( theCanvas ).withBorderWidth( 1 ).withBorderChar( ' ' ).toStrings();
		}
		theCanvas = new TextBorderBuilder().withTextBoxGrid( _textBoxGrid ).withText( theCanvas ).withBoxBorderMode( BoxBorderMode.ALL ).toStrings();

		if ( _isEscCodesEnabled ) {
			theCanvas[0] = _bannerBorderEscCode + theCanvas[0] + _resetEscCode;
			if ( theCanvas.length > 1 ) {
				for ( int i = 1; i < theCanvas.length - 1; i++ ) {
					theCanvas[i] = _bannerBorderEscCode + theCanvas[i].substring( 0, 1 ) + _resetEscCode + _bannerEscCode + theCanvas[i].substring( 1, theCanvas[i].length() - 1 ) + _resetEscCode + _bannerBorderEscCode + theCanvas[i].substring( theCanvas[i].length() - 1 ) + _resetEscCode;
				}
			}
			theCanvas[theCanvas.length - 1] = _bannerBorderEscCode + theCanvas[theCanvas.length - 1] + _resetEscCode;
		}
		_stdStream.print( fromTextBlock( theCanvas, toLineBreak() ) );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printCopyrightNote() {
		_stdStream.println( _copyrightNote );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printDescription() {
		_stdStream.println( (_isEscCodesEnabled ? _descriptionEscCode : "") + _description + (_isEscCodesEnabled ? _resetEscCode : "") );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printLicenseNote() {
		_stdStream.println( _licenseNote );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printLn() {
		_stdStream.println();
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printLn( String aLine ) {
		String[] theLines = new TextBlockBuilder().withText( aLine ).withColumnWidth( _consoleWidth ).withSplitTextMode( SplitTextMode.AT_SPACE ).toStrings();
		_stdStream.print( fromTextBlock( theLines, toLineBreak() ) );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printOptions() {
		if ( _rootArgsSyntax != null ) {
			List<? extends Operand<?>> theOperands = _rootArgsSyntax.toOperands();
			Map<String, String[]> theOptArgs = new HashMap<>();
			int theMaxLength = 0;
			String eOpt;
			int eEffectiveLength;
			for ( Operand<?> eOperand : theOperands ) {
				eOpt = CliUtility.toSpec( eOperand, _isEscCodesEnabled ? _optEscCode : null, _isEscCodesEnabled ? _paramEscCode : null, _isEscCodesEnabled ? _resetEscCode : null );
				eEffectiveLength = AnsiEscapeCode.toLength( eOpt, _isEscCodesEnabled );
				if ( eEffectiveLength > theMaxLength ) theMaxLength = eEffectiveLength;
				theOptArgs.put( eOpt, new String[] { eOpt, ":", " ", eOperand.getDescription() } );
			}
			TableBuilder theTable = new TableBuilder().withTableStyle( TableStyle.BLANK_HEADER_BLANK_BODY ).withRowWidth( _consoleWidth ).withPrintStream( _stdStream ).withEscapeCodesEnabled( _isEscCodesEnabled ).withLeftBorder( false ).withRightBorder( false ).withDividerLine( false );
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withColumnWidth( theMaxLength, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( 1, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( 1, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.LEFT ).withColumnSplitTextMode( SplitTextMode.AT_SPACE );
			if ( _isEscCodesEnabled ) {
				theTable.withRowColumnEscapeCode( _descriptionEscCode );
			}
			theTable.withLineBreak( toLineBreak() );
			List<String> theKeys = new ArrayList<>( theOptArgs.keySet() );
			Collections.sort( theKeys, new Comparator<String>() {
				@Override
				public int compare( String a, String b ) {
					if ( a != null ) while ( a.startsWith( "-" ) ) a = a.substring( 1 );
					if ( b != null ) while ( b.startsWith( "-" ) ) b = b.substring( 1 );
					return a.compareToIgnoreCase( b );
				}
			} );
			for ( String eKey : theKeys ) {
				theTable.printRowContinue( theOptArgs.get( eKey ) );
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printExamples() {
		if ( _examples != null && _examples.size() != 0 ) {
			int theMaxLength = 0;
			String eDescription;
			for ( Example eUsage : _examples ) {
				eDescription = eUsage._description;
				if ( eDescription.length() > theMaxLength ) theMaxLength = eDescription.length();
			}
			if ( theMaxLength > _consoleWidth / 2 ) {
				theMaxLength = _consoleWidth / 2;
				if ( theMaxLength < ConsoleDimension.MIN_WIDTH.getValue() / 2 ) {
					theMaxLength = ConsoleDimension.MIN_WIDTH.getValue() / 2;
				}
			}

			TableBuilder theTable = new TableBuilder().withTableStyle( TableStyle.BLANK_HEADER_BLANK_BODY ).withRowWidth( _consoleWidth ).withPrintStream( _stdStream ).withEscapeCodesEnabled( _isEscCodesEnabled ).withLeftBorder( false ).withRightBorder( false ).withDividerLine( false );
			if ( _isEscCodesEnabled ) {
				theTable.setResetEscapeCode( _resetEscCode );
			}
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withColumnWidth( theMaxLength, ColumnWidthType.ABSOLUTE );
			if ( _isEscCodesEnabled ) {
				theTable.withRowColumnEscapeCode( _descriptionEscCode );
			}
			theTable.addColumn().withColumnWidth( 2, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( _name.length(), ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnWidth( 1, ColumnWidthType.ABSOLUTE );
			theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.LEFT ).withColumnSplitTextMode( SplitTextMode.AT_SPACE );
			theTable.withLineBreak( toLineBreak() );

			String theCommand = (_isEscCodesEnabled ? _commandEscCode : "") + _name + (_isEscCodesEnabled ? _resetEscCode : "");
			for ( Example eUsage : _examples ) {
				String eArgs = "";
				String eOpt;
				for ( Operand<?> eOperand : eUsage.getOperands() ) {
					if ( eArgs.length() != 0 ) eArgs += " ";
					eOpt = CliUtility.toShortSpec( eOperand, _isEscCodesEnabled ? _optEscCode : null, _isEscCodesEnabled ? _paramEscCode : null, _isEscCodesEnabled ? _resetEscCode : null );
					eArgs += eOpt;
				}
				theTable.printRowContinue( eUsage.getDescription(), ": ", theCommand, " ", eArgs );
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printSeparatorLn() {
		String theLine = new TextLineBuilder().withColumnWidth( _consoleWidth ).withLineChar( _separatorLnChar ).toString() + toLineBreak();
		if ( _isEscCodesEnabled ) {
			theLine = _lineSeparatorEscCode + theLine + _resetEscCode;
		}
		_stdStream.print( theLine );
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printSynopsis() {
		List<Syntaxable> theChildren;
		if ( _rootArgsSyntax instanceof XorCondition ) {
			XorCondition theCases = (XorCondition) _rootArgsSyntax;
			theChildren = theCases.getChildren();
		}
		else {
			theChildren = new ArrayList<>();
			theChildren.add( _rootArgsSyntax );
		}
		TableBuilder theTable = new TableBuilder().withTableStyle( TableStyle.BLANK_HEADER_BLANK_BODY ).withRowWidth( _consoleWidth ).withPrintStream( _stdStream ).withEscapeCodesEnabled( _isEscCodesEnabled ).withLeftBorder( false ).withRightBorder( false ).withDividerLine( true );
		if ( _isEscCodesEnabled ) {
			theTable.setResetEscapeCode( _resetEscCode );
		}
		theTable.addColumn().withColumnWidth( _name.length(), ColumnWidthType.ABSOLUTE ).withColumnHorizAlignTextMode( HorizAlignTextMode.RIGHT );
		theTable.addColumn().withColumnHorizAlignTextMode( HorizAlignTextMode.LEFT ).withColumnSplitTextMode( SplitTextMode.AT_SPACE );
		theTable.withLineBreak( toLineBreak() );

		String theCommand = (_isEscCodesEnabled ? _commandEscCode : "") + _name + (_isEscCodesEnabled ? _resetEscCode : "");
		String eUsage;
		for ( Syntaxable eChild : theChildren ) {
			eUsage = (eChild != null) ? (_isEscCodesEnabled ? eChild.toSynopsis( _syntaxNotation, _optEscCode, _paramEscCode, _resetEscCode ) : eChild.toSynopsis( _syntaxNotation )) : "<?>";
			theTable.printRowContinue( theCommand, eUsage );
		}
		_stdStream.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		if ( _rootArgsSyntax != null ) {
			_rootArgsSyntax.reset();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerFont( Font aBannerFont ) {
		_bannerFont = aBannerFont;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBannerFontPalette( char[] aColorPalette ) {
		_bannerFontPalette = aColorPalette;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConsoleWidth( int aConsoleWidth ) {
		_consoleWidth = _maxConsoleWidth != -1 ? (_maxConsoleWidth < aConsoleWidth ? _maxConsoleWidth : aConsoleWidth) : aConsoleWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCopyrightNote( String aCopyrightNote ) {
		_copyrightNote = aCopyrightNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription( String aDescription ) {
		_description = aDescription;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setErrorOut( PrintStream aErrorOut ) {
		_errStream = aErrorOut;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEscapeCodesEnabled() {
		return _isEscCodesEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEscapeCodesEnabled( boolean isEscCodeEnabled ) {
		_isEscCodesEnabled = isEscCodeEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescriptionEscapeCode() {
		return _descriptionEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescriptionEscapeCode( String aDescriptionEscCode ) {
		_descriptionEscCode = aDescriptionEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getCommandEscapeCode() {
		return _commandEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCommandEscapeCode( String aCommandEscCode ) {
		_commandEscCode = aCommandEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLicenseNote( String aLicenseNote ) {
		_licenseNote = aLicenseNote;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLineBreak( String aLineBreak ) {
		if ( aLineBreak == null ) {
			aLineBreak = Terminal.getLineBreak();
		}
		_lineBreak = aLineBreak;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMaxConsoleWidth( int aMaxConsoleWidth ) {
		_maxConsoleWidth = aMaxConsoleWidth;
		_consoleWidth = _maxConsoleWidth != -1 ? (_maxConsoleWidth < _consoleWidth ? _maxConsoleWidth : _consoleWidth) : _consoleWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName( String aName ) {
		_name = aName;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSeparatorLnChar( char aSeparatorLnChar ) {
		_separatorLnChar = aSeparatorLnChar;
		_hasOverrideSeparatorLnChar = true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public char getSeparatorLnChar() {
		return _separatorLnChar;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setStandardOut( PrintStream aStandardOut ) {
		_stdStream = aStandardOut;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSyntaxNotation( SyntaxNotation aSyntaxNotation ) {
		_syntaxNotation = aSyntaxNotation;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTitle( String aTitle ) {
		_title = aTitle;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextBoxGrid getTextBoxGrid() {
		return _textBoxGrid;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTextBoxGrid( TextBoxGrid aTextBoxGrid ) {
		_textBoxGrid = aTextBoxGrid;
		if ( !_hasOverrideSeparatorLnChar ) {
			_separatorLnChar = aTextBoxGrid.getInnerLine();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Example> getExamples() {
		return _examples;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addExample( Example aExampleUsage ) {
		if ( _examples == null ) {
			synchronized ( this ) {
				if ( _examples == null ) {
					_examples = new ArrayList<Example>();
				}
			}
		}
		_examples.add( aExampleUsage );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Heuristically loads the arguments without any syntax required, e.g.
	 * without any root {@link ArgsSyntax} to be set.
	 * 
	 * @param aArgs The arguments to be loaded.
	 * @param aDelimiter The delimiter to resolve name clashes.
	 * 
	 * @return A list of heuristically determined {@link Flag} and
	 *         {@link StringOperand} instances.
	 */
	protected static List<? extends Operand<?>> fromArgs( String[] aArgs, char aDelimiter ) {
		List<Operand<?>> theOperands = new ArrayList<>();
		Map<String, String> theArgs = RuntimeUtility.toProperties( aArgs, CommandArgPrefix.toPrefixes(), aDelimiter );
		for ( String eKey : theArgs.keySet() ) {
			if ( Literal.TRUE.getValue().equalsIgnoreCase( theArgs.get( eKey ) ) ) {
				theOperands.add( new Flag( eKey, true ) );
			}
			else {
				theOperands.add( new StringOperand( new PropertyImpl( eKey, theArgs.get( eKey ) ) ) );
			}
		}
		return theOperands;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To line break.
	 *
	 * @return the string
	 */
	private String toLineBreak() {
		if ( Terminal.isLineBreakRequired( _consoleWidth ) ) {
			return _lineBreak;
		}
		return "";
	}

	private boolean hasBorder( String[] aCanvas ) {
		if ( aCanvas != null && aCanvas.length != 0 ) {
			char ePrevChar = aCanvas[0].length() != 0 ? aCanvas[0].charAt( 0 ) : ' ';
			// Top |-->
			for ( int i = 0; i < aCanvas[0].length(); i++ ) {
				if ( ePrevChar != aCanvas[0].charAt( i ) ) {
					return false;
				}
				ePrevChar = aCanvas[0].charAt( i );
			}
			// Bottom |-->
			for ( int i = 0; i < aCanvas[aCanvas.length - 1].length(); i++ ) {
				if ( ePrevChar != aCanvas[aCanvas.length - 1].charAt( i ) ) {
					return false;
				}
				ePrevChar = aCanvas[aCanvas.length - 1].charAt( i );
			}
			// Left |-->
			for ( int i = 0; i < aCanvas.length; i++ ) {
				if ( ePrevChar != aCanvas[i].charAt( 0 ) ) {
					return false;
				}
				ePrevChar = aCanvas[i].charAt( 0 );
			}
			// Right |-->
			for ( int i = 0; i < aCanvas.length; i++ ) {
				if ( ePrevChar != aCanvas[i].charAt( aCanvas[i].length() - 1 ) ) {
					return false;
				}
				ePrevChar = aCanvas[i].charAt( aCanvas[i].length() - 1 );
			}
		}
		return true;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
