package com.github.azbh111.utils.java.javascript;

import com.github.azbh111.utils.java.resource.ResourceUtils;
import com.github.azbh111.utils.java.stack.StackTraceUtils;

import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

/**
 *
 * @author pyz
 * @date 2019/4/29 8:50 PM
 */
public class JavaScriptUtils {

    /**
     * 从classpath中加载js脚本
     *
     * @param path
     * @return
     * @throws IOException
     * @throws ScriptException
     */
    public static Invocable loadScript(String path) throws IOException, ScriptException {
        URL in = ResourceUtils.getResource(StackTraceUtils.getCallerClass(), path);
        if (in == null) {
            return null;
        }
        try (InputStream is = in.openConnection().getInputStream()) {
            return resolveScript(new HashMap<>(), engine -> engine.eval(new InputStreamReader(is, StandardCharsets.UTF_8)));
        }
    }

    /**
     * 从classpath中加载js脚本
     *
     * @param path
     * @param bindings
     * @return
     * @throws IOException
     * @throws ScriptException
     */
    public static Invocable loadScript(String path, Map<String, Object> bindings) throws IOException, ScriptException {
        URL in = ResourceUtils.getResource(StackTraceUtils.getCallerClass(), path);
        if (in == null) {
            return null;
        }
        try (InputStream is = in.openConnection().getInputStream()) {
            return resolveScript(bindings, engine -> engine.eval(new InputStreamReader(is, StandardCharsets.UTF_8)));
        }
    }

    /**
     * 解析js脚本
     *
     * @param script
     * @return
     * @throws IOException
     * @throws ScriptException
     */
    public static Invocable resolveScript(String script) throws IOException, ScriptException {
        return resolveScript(script, null);
    }

    /**
     * 解析js脚本
     *
     * @param script
     * @param bindings
     * @return
     * @throws IOException
     * @throws ScriptException
     */
    public static Invocable resolveScript(String script, Map<String, Object> bindings) throws IOException, ScriptException {
        return resolveScript(bindings, engine -> engine.eval(script));
    }

    /**
     * 从classpath中加载js脚本
     *
     * @param bindings
     * @return
     * @throws IOException
     * @throws ScriptException
     */
    private static Invocable resolveScript(Map<String, Object> bindings, ScriptEngineEval eval) throws IOException, ScriptException {
        ScriptEngineManager m = new ScriptEngineManager();
        ScriptEngine engine = m.getEngineByName("JavaScript");
        if (bindings != null && !bindings.isEmpty()) {
            bindings.forEach(engine::put);
        }
        eval.accept(engine);
        return (Invocable) engine;
    }

    interface ScriptEngineEval {
        void accept(ScriptEngine engine) throws ScriptException;
    }

}
