package com.github.azbh111.utils.java.net;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.*;

/**
 * @author: zyp
 * @since: 2021/12/28 上午10:35
 */
public class NetUtils {

    /**
     * 从网卡上读取本机拥有的ip
     *
     * @param
     * @return com.github.azbh111.utils.java.net.Ips
     * @author zhengyongpan
     * @since 2021/12/28 下午12:39
     */
    public static Ips getIp() throws SocketException {
        Map<String, List<String>> localIps = new LinkedHashMap<>();
        Map<String, List<String>> netIps = new LinkedHashMap<>();
        Enumeration<NetworkInterface> netInterfaces = NetworkInterface.getNetworkInterfaces();
        InetAddress inetAddress = null;
        boolean finded = false;// 是否找到外网IP
        while (netInterfaces.hasMoreElements() && !finded) {
            NetworkInterface ni = netInterfaces.nextElement();
            Enumeration<InetAddress> address = ni.getInetAddresses();
            while (address.hasMoreElements()) {
                inetAddress = address.nextElement();
                if (inetAddress.getAddress().length != 4) {
                    continue;
                }
                if (NetUtils.isInnerIp(inetAddress.getAddress())) {
                    localIps.computeIfAbsent(ni.getName(), k -> new ArrayList<>())
                            .add(inetAddress.getHostAddress());
                } else {
                    netIps.computeIfAbsent(ni.getName(), k -> new ArrayList<>())
                            .add(inetAddress.getHostAddress());
                }
            }
        }
        return new Ips(localIps, netIps);
    }

    private static final List<IpMaskFilter> INNER_IP_MASK_FILTERS;

    static {
        INNER_IP_MASK_FILTERS = new ArrayList<>();
        INNER_IP_MASK_FILTERS.add(new IpMaskFilter(toIpByte(new int[]{255, 0, 0, 0}), toIpByte(new int[]{10, 0, 0, 0})));
        INNER_IP_MASK_FILTERS.add(new IpMaskFilter(toIpByte(new int[]{255, 255, 0, 0}), toIpByte(new int[]{172, 16, 0, 0})));
        INNER_IP_MASK_FILTERS.add(new IpMaskFilter(toIpByte(new int[]{255, 255, 255, 255}), toIpByte(new int[]{192, 166, 0, 0})));
        INNER_IP_MASK_FILTERS.add(new IpMaskFilter(toIpByte(new int[]{255, 255, 255, 255}), toIpByte(new int[]{127, 0, 0, 1})));
        INNER_IP_MASK_FILTERS.add(new IpMaskFilter(toIpByte(new int[]{255, 255, 255, 255}), toIpByte(new int[]{0, 0, 0, 0})));
    }

    /**
     * 判断内网ip
     *
     * @param ip ip
     * @return boolean
     * @author zhengyongpan
     * @since 2021/12/28 上午11:18
     */
    public static boolean isInnerIp(String ip) {
        if (ip == null) {
            return false;
        }
        for (IpMaskFilter ipMaskFilter : INNER_IP_MASK_FILTERS) {
            if (ipMaskFilter.match(ip)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断内网ip
     *
     * @param ip ip
     * @return boolean
     * @author zhengyongpan
     * @since 2021/12/28 上午11:18
     */
    public static boolean isInnerIp(byte[] ip) {
        if (ip == null) {
            return false;
        }
        for (IpMaskFilter ipMaskFilter : INNER_IP_MASK_FILTERS) {
            if (ipMaskFilter.match(ip)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 将byte格式的ip转换成可读的字符串
     *
     * @param ip ip
     * @return java.lang.String
     * @author zhengyongpan
     * @since 2021/12/28 下午12:39
     */
    public static String toIpString(byte[] ip) {
        validate(ip);
        StringBuilder sb = new StringBuilder(3 * 4 + 3);
        sb.append(toIpInt(ip[0]));
        sb.append('.');
        sb.append(toIpInt(ip[1]));
        sb.append('.');
        sb.append(toIpInt(ip[2]));
        sb.append('.');
        sb.append(toIpInt(ip[3]));
        return sb.toString();
    }

    /**
     * 将计算机使用的byte值转换成人类可读的ip值
     *
     * @param value value
     * @return int
     * @author zhengyongpan
     * @since 2021/12/28 下午12:39
     */
    public static int toIpInt(byte value) {
        return value < 0 ? value + 256 : value;
    }

    /**
     * 将计算机使用的byte值转换成人类可读的ip值
     *
     * @param ip ip
     * @return int[]
     * @author zhengyongpan
     * @since 2021/12/28 下午12:38
     */
    public static int[] toIpInt(byte[] ip) {
        int[] val = new int[4];
        for (int i = 0; i < 4; i++) {
            val[i] = toIpInt(ip[i]);
        }
        return val;
    }

    /**
     * 将人类可读的ip值转换成计算机使用的byte值
     *
     * @param value value
     * @return byte
     * @author zhengyongpan
     * @since 2021/12/28 下午12:38
     */
    public static byte toIpByte(int value) {
        return (byte) (value > 127 ? value - 256 : value);
    }

    /**
     * 将人类可读的ip值转换成计算机使用的byte值
     *
     * @param ip ip
     * @return byte[]
     * @author zhengyongpan
     * @since 2021/12/28 下午12:38
     */
    public static byte[] toIpByte(int[] ip) {
        byte[] val = new byte[4];
        for (int i = 0; i < 4; i++) {
            val[i] = toIpByte(ip[i]);
        }
        return val;
    }

    public static void validate(byte[] ip) {
        if (ip == null) {
            throw new NullPointerException();
        }
        if (ip.length != 4) {
            throw new RuntimeException("length of ip not equals 4");
        }
    }
}
