package com.github.rameshdev.httputils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.HttpURLConnection;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import lombok.Data;

/**
 * The type Url fetcher.
 */
@Data
public class UrlFetcher {

    private final Map<String, String> defaultHeaders = new HashMap<>();

    private final FetchConfig fetchConfig;

    /**
     * Instantiates a new Url fetcher.
     */
    public UrlFetcher() {
        this(FetchConfig.withDefaults());
    }

    /**
     * Instantiates a new Url fetcher.
     *
     * @param fetchConfig the fetch config
     */
    public UrlFetcher(FetchConfig fetchConfig) {
        this.fetchConfig = fetchConfig;
    }

    /**
     * Add header.
     *
     * @param key   the key
     * @param value the value
     */
    public void addHeader(String key, String value) {

        if (key != null & value != null)
            this.defaultHeaders.put(key, value);
    }

    /**
     * Add all headers.
     *
     * @param headers the headers
     */
    public void addAllHeaders(Map<String, String> headers) {

        if (headers != null && !headers.isEmpty())
            defaultHeaders.putAll(headers);
    }

    /**
     * Make get request.
     *
     * @param url the url string
     * @return the http response
     * @throws IOException the iO exception
     */
    public HttpResponse get(String url) throws IOException {
        return request(HttpRequest.builder().method(HttpMethod.GET).setUrl(url).build());
    }

    /**
     * Post http response.
     *
     * @param url         the url
     * @param contentType the content type
     * @param payload     the payload
     * @return the http response
     * @throws IOException the io exception
     */
    public HttpResponse post(String url, String contentType, byte[] payload) throws IOException {
        return request(HttpMethod.POST, url, contentType, payload);
    }

    /**
     * Put http response.
     *
     * @param url         the url
     * @param contentType the content type
     * @param payload     the payload
     * @return the http response
     * @throws IOException the io exception
     */
    public HttpResponse put(String url, String contentType, byte[] payload) throws IOException {
        return request(HttpMethod.PUT, url, contentType, payload);
    }

    /**
     * Make get request.
     *
     * @param method      the method
     * @param url         the url string
     * @param contentType the content type
     * @param payload     the payload
     * @return the http response
     * @throws IOException the iO exception
     */
    public HttpResponse request(HttpMethod method, String url, String contentType, byte[] payload) throws IOException {

        HttpRequest request = HttpRequest.builder()
                .method(method)
                .setUrl(url)
                .contentType(contentType)
                .payload(payload)
                .build();
        return request(request);
    }

    /**
     * Make request.
     *
     * @param request the request
     * @return the http response
     * @throws IOException the iO exception
     */
    public HttpResponse request(HttpRequest request) throws IOException {

        HttpURLConnection conn = (HttpURLConnection) request.getUrl().openConnection();

        conn.setRequestMethod(request.getMethod().toString());

        FetchConfig config = request.getFetchConfig();
        if (config == null)
            config = this.fetchConfig;

        if (config.getConnectionTimeOutSecs() > 0)
            conn.setConnectTimeout(config.getConnectionTimeOutSecs() * 1000);

        conn.setInstanceFollowRedirects(config.isFollowRedirects());

        attachHeaders(conn, request.getHeaders());

        // attach payload
        if (request.getMethod() != HttpMethod.GET) {

            conn.setRequestProperty("Content-Type", request.getContentType());

            if (request.getPayload() != null) {
                conn.setDoOutput(true);

                OutputStream os = conn.getOutputStream();
                os.write(request.getPayload());
                os.flush();
            }
        }

        InputStream stream = getInputStream(conn);
        String responseContent = streamToString(stream);

        HttpResponse httpResponse = new HttpResponse(conn.getResponseCode(), responseContent);
        httpResponse.setHeaders(conn.getHeaderFields());

        return httpResponse;
    }

    private void attachHeaders(HttpURLConnection conn, Map<String, String> requestHeaders) {

        Map<String, String> headers;

        if (requestHeaders != null && !requestHeaders.isEmpty()) {
            headers = new HashMap<>();
            headers.putAll(defaultHeaders);
            headers.putAll(requestHeaders);
        } else {
            headers = defaultHeaders;
        }

        for (Map.Entry<String, String> header : headers.entrySet()) {
            conn.setRequestProperty(header.getKey(), header.getValue());
        }

        if (!headers.containsKey("user-agent") && !headers.containsKey("User-Agent")) {
            String agent = UserAgentHelper.getUserAgent();
            if (agent != null) {
                conn.setRequestProperty("User-Agent", agent);
            }
        }

    }

    /**
     * Stream to string string.
     *
     * @param is the is
     * @return the string
     * @throws IOException the io exception
     */
    public String streamToString(InputStream is) throws IOException {
        if (is == null)
            return null;

        BufferedReader rd = new BufferedReader(new InputStreamReader(is));
        StringBuilder sb = new StringBuilder();
        String line;
        while ((line = rd.readLine()) != null) {
            sb.append(line);
        }
        rd.close();
        return sb.toString();
    }

    /**
     * Gets input stream.
     *
     * @param conn the conn
     * @return the input stream
     * @throws IOException the io exception
     */
    public InputStream getInputStream(HttpURLConnection conn) throws IOException {

        InputStream stream = null;
        try {
            stream = conn.getInputStream();
        } catch (IOException e) {
            if (conn.getResponseCode() != 200) {
                stream = conn.getErrorStream();
            }
        }
        return stream;
    }

    static {
        allowMethods("PATCH", "HEAD");
    }

    private static void allowMethods(String... methods) {
        try {
            Field methodsField = HttpURLConnection.class.getDeclaredField("methods");

            Field modifiersField = Field.class.getDeclaredField("modifiers");
            modifiersField.setAccessible(true);
            modifiersField.setInt(methodsField, methodsField.getModifiers() & ~Modifier.FINAL);

            methodsField.setAccessible(true);

            String[] oldMethods = (String[]) methodsField.get(null);
            Set<String> methodsSet = new LinkedHashSet<>(Arrays.asList(oldMethods));
            methodsSet.addAll(Arrays.asList(methods));
            String[] newMethods = methodsSet.toArray(new String[0]);

            methodsField.set(null/*static field*/, newMethods);
        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new IllegalStateException(e);
        }
    }
}
