/*
 * Decompiled with CFR 0.152.
 */
package com.github.tadukoo.java.testing.util;

import com.github.tadukoo.java.Visibility;
import com.github.tadukoo.java.annotation.EditableJavaAnnotation;
import com.github.tadukoo.java.javaclass.EditableJavaClass;
import com.github.tadukoo.java.javadoc.EditableJavadoc;
import com.github.tadukoo.java.method.EditableJavaMethod;
import com.github.tadukoo.java.testing.JavaClassParsingTest;

public class FileUtilTest
extends JavaClassParsingTest {
    public FileUtilTest() {
        super("package com.github.tadukoo.util;\n\nimport java.io.BufferedReader;\nimport java.io.BufferedWriter;\nimport java.io.File;\nimport java.io.FileInputStream;\nimport java.io.FileNotFoundException;\nimport java.io.FileOutputStream;\nimport java.io.FileReader;\nimport java.io.FileWriter;\nimport java.io.IOException;\nimport java.io.Reader;\nimport java.io.Writer;\nimport java.nio.file.Files;\nimport java.nio.file.Path;\nimport java.nio.file.Paths;\nimport java.util.ArrayList;\nimport java.util.Collection;\nimport java.util.Comparator;\nimport java.util.List;\nimport java.util.stream.Collectors;\nimport java.util.stream.Stream;\nimport java.util.zip.ZipEntry;\nimport java.util.zip.ZipInputStream;\nimport java.util.zip.ZipOutputStream;\n\n/**\n * Util functions for dealing with Files.\n * \n * @author Logan Ferree (Tadukoo)\n * @version Beta v.0.5.2\n * @since Pre-Alpha\n */\npublic final class FileUtil{\n\t\n\t/** Not allowed to create a FileUtil */\n\tprivate FileUtil(){ }\n\t\n\t/**\n\t * Finds the file extension for the given filepath, if one exists.\n\t * \n\t * @param filepath The filepath to grab the extension for\n\t * @return The found file extension, or null\n\t */\n\tpublic static String getFileExtension(String filepath){\n\t\t// Find the last dot in the filepath\n\t\tint lastDotIndex = filepath.lastIndexOf('.');\n\t\t\n\t\t// If one isn't found, return null\n\t\tif(lastDotIndex == -1){\n\t\t\treturn null;\n\t\t}else{\n\t\t\t// If we found a dot, return whatever's after it\n\t\t\treturn filepath.substring(lastDotIndex + 1);\n\t\t}\n\t}\n\t\n\t/**\n\t * Creates a List of all Files in the given directory and any of\n\t * its subdirectories.\n\t * \n\t * @param directoryPath The path to the directory to check\n\t * @return A List of all Files in the directory and its subdirectories\n\t * @throws IOException If something goes wrong in listing the files\n\t */\n\tpublic static List<File> listAllFiles(Path directoryPath) throws IOException{\n\t\ttry(Stream<Path> pathStream = Files.walk(directoryPath)){\n\t\t\treturn pathStream\n\t\t\t\t\t.filter(Files::isRegularFile)\n\t\t\t\t\t.map(Path::toFile)\n\t\t\t\t\t.collect(Collectors.toList());\n\t\t}\n\t}\n\t\n\t/**\n\t * Creates a List of all Files in the given directory and any of its\n\t * subdirectories.\n\t * \n\t * @param directory The directory (as a File) to check\n\t * @return A List of all Files in the directory and its subdirectories\n\t * @throws IOException If something goes wrong in listing the files\n\t */\n\tpublic static List<File> listAllFiles(File directory) throws IOException{\n\t\treturn listAllFiles(directory.toPath());\n\t}\n\t\n\t/**\n\t * Creates a List of all Files in the given directory and any of its\n\t * subdirectories.\n\t * \n\t * @param directory The directory path (as a String) to check\n\t * @return A List of all Files in the directory and its subdirectories\n\t * @throws IOException If something goes wrong in listing the files\n\t */\n\tpublic static List<File> listAllFiles(String directory) throws IOException{\n\t\treturn listAllFiles(Paths.get(directory));\n\t}\n\t\n\t/**\n\t * Creates a List of all files in the given directory and any of\n\t * its subdirectories. The files are returned as Paths.\n\t * \n\t * @param directory The path to the directory to check\n\t * @return A List of all files in the directory and its subdirectories as Paths\n\t * @throws IOException If something goes wrong in listing the files\n\t */\n\tpublic static List<Path> listAllFilesAsPaths(Path directory) throws IOException{\n\t\ttry(Stream<Path> pathStream = Files.walk(directory)){\n\t\t\treturn pathStream\n\t\t\t\t\t.filter(Files::isRegularFile)\n\t\t\t\t\t.collect(Collectors.toList());\n\t\t}\n\t}\n\t\n\t/**\n\t * Creates a List of all files in the given directory and any of its\n\t * subdirectories. The files are returned as Paths.\n\t * \n\t * @param directory The directory (as a File) to check\n\t * @return A List of all files in the directory and its subdirectories as Paths\n\t * @throws IOException If something goes wrong in listing the files\n\t */\n\tpublic static List<Path> listAllFilesAsPaths(File directory) throws IOException{\n\t\treturn listAllFilesAsPaths(directory.toPath());\n\t}\n\t\n\t/**\n\t * Creates a List of all files in the given directory and any of its\n\t * subdirectories. The files are returned as Paths.\n\t * \n\t * @param directory The directory path (as a String) to check\n\t * @return A List of all files in the directory and its subdirectories as Paths\n\t * @throws IOException If something goes wrong in listing the files\n\t */\n\tpublic static List<Path> listAllFilesAsPaths(String directory) throws IOException{\n\t\treturn listAllFilesAsPaths(Paths.get(directory));\n\t}\n\t\n\t/**\n\t * Creates a file at the given filepath, including any directories necessary,\n\t * and returns the {@link File} object to be used.\n\t * \n\t * @param filepath The path for the File to be created\n\t * @return The newly created File\n\t * @throws IOException If something goes wrong in creating the file\n\t */\n\t@SuppressWarnings(\"ResultOfMethodCallIgnored\")\n\tpublic static File createFile(String filepath) throws IOException{\n\t\t// Create a File object from the given filepath\n\t\tFile file = new File(filepath);\n\t\t\n\t\t// If a directory is specified, create it if it doesn't exist\n\t\tFile parentFile = file.getParentFile();\n\t\tif(parentFile != null && !parentFile.exists()){\n\t\t\tfile.getParentFile().mkdirs();\n\t\t}\n\t\t\n\t\t// If the file doesn't exist, create it\n\t\tif(!file.exists()){\n\t\t\tfile.createNewFile();\n\t\t}\n\t\t\n\t\treturn file;\n\t}\n\t\n\t/**\n\t * Deletes the file at the given filepath\n\t * \n\t * @param filepath The path for the File to be deleted\n\t * @return If the file was deleted or not\n\t */\n\tpublic static boolean deleteFile(String filepath){\n\t\t// Create the File object from the given filepath\n\t\tFile file = new File(filepath);\n\t\t\n\t\treturn file.delete();\n\t}\n\t\n\t/**\n\t * Creates a directory at the given directoryPath, including any parent directories\n\t * necessary, and returns the {@link File} object to be used.\n\t * \n\t * @param directoryPath The path for the directory to be created\n\t * @return The newly created directory\n\t */\n\tpublic static File createDirectory(String directoryPath){\n\t\t// Create a File object from the given filepath\n\t\tFile directory = new File(directoryPath);\n\t\t\n\t\t// Make the directory and its parent folders\n\t\tif(!directory.mkdirs() && !directory.exists()){\n\t\t\tthrow new IllegalStateException(\"Failed to create folder: \" + directoryPath + \"!\");\n\t\t}\n\t\t\n\t\treturn directory;\n\t}\n\t\n\t/**\n\t * Deletes the directory at the given directoryPath, including any files contained\n\t * within it.\n\t * \n\t * @param directoryPath The path to the directory to be deleted\n\t * @throws IOException If anything goes wrong\n\t */\n\tpublic static void deleteDirectory(String directoryPath) throws IOException{\n\t\ttry(Stream<Path> pathStream = Files.walk(Path.of(directoryPath))){\n\t\t\t//noinspection ResultOfMethodCallIgnored\n\t\t\tpathStream\n\t\t\t\t\t.sorted(Comparator.reverseOrder())\n\t\t\t\t\t.map(Path::toFile)\n\t\t\t\t\t.forEach(File::delete);\n\t\t}\n\t}\n\t\n\t/**\n\t * Creates a new {@link BufferedReader} for the file at the given filepath.\n\t * \n\t * @param filepath The path of the file to be read\n\t * @return A {@link BufferedReader} for the given file\n\t * @throws FileNotFoundException If the file can't be found\n\t */\n\tpublic static BufferedReader setupFileReader(String filepath) throws FileNotFoundException{\n\t\treturn new BufferedReader(new FileReader(filepath));\n\t}\n\t\n\t/**\n\t * Creates a new {@link BufferedReader} for the given {@link File}.\n\t * \n\t * @param file The {@link File} to be read\n\t * @return A {@link BufferedReader} for the given {@link File}\n\t * @throws FileNotFoundException If the file can't be found\n\t */\n\tpublic static BufferedReader setupFileReader(File file) throws FileNotFoundException{\n\t\treturn new BufferedReader(new FileReader(file));\n\t}\n\t\n\t/**\n\t * Checks if the file at the given filepath exists.\n\t * \n\t * @param filepath The path of the file to be checked\n\t * @return true if the file exists, false if it doesn't\n\t */\n\tpublic static boolean exists(String filepath){\n\t\treturn Files.exists(Paths.get(filepath));\n\t}\n\t\n\t/**\n\t * Checks if the given {@link File} exists.\n\t * \n\t * @param file The {@link File} to be checked\n\t * @return true if the file exists, false if it doesn't\n\t */\n\tpublic static boolean exists(File file){\n\t\treturn Files.exists(file.toPath());\n\t}\n\t\n\t/**\n\t * Checks if the file at the given filepath does not exist.\n\t * \n\t * @param filepath The path of the file to be checked\n\t * @return false if the file exists, true if it doesn't\n\t */\n\tpublic static boolean notExists(String filepath){\n\t\treturn Files.notExists(Paths.get(filepath));\n\t}\n\t\n\t/**\n\t * Checks if the given {@link File} does not exist.\n\t * \n\t * @param file The {@link File} to be checked\n\t * @return false if the file exists, true if it doesn't\n\t */\n\tpublic static boolean notExists(File file){\n\t\treturn Files.notExists(file.toPath());\n\t}\n\t\n\t/**\n\t * Reads the file at the given filepath as a String\n\t * \n\t * @param filepath The path of the file to be read\n\t * @return A String representing the contents of the file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static String readAsString(String filepath) throws IOException{\n\t\treturn Files.readString(Paths.get(filepath));\n\t}\n\t\n\t/**\n\t * Reads the given {@link File} as a String\n\t * \n\t * @param file The {@link File} to be read\n\t * @return A String representing the contents of the file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static String readAsString(File file) throws IOException{\n\t\treturn Files.readString(file.toPath());\n\t}\n\t\n\t/**\n\t * Creates a List of Strings for each line in the file being read from the given filepath.\n\t * \n\t * @param filepath The path to the file to be read\n\t * @return A List of lines in the file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static List<String> readLinesAsList(String filepath) throws IOException{\n\t\treturn Files.readAllLines(Paths.get(filepath));\n\t}\n\t\n\t/**\n\t * Creates a List of Strings for each line in the file being read.\n\t * \n\t * @param file The {@link File} to read\n\t * @return A List of lines in the file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static List<String> readLinesAsList(File file) throws IOException{\n\t\treturn Files.readAllLines(file.toPath());\n\t}\n\t\n\t/**\n\t * Creates a List of Strings for each line in the file being read in the\n\t * given {@link Reader}.\n\t * \n\t * @param reader The Reader to use in reading\n\t * @return A List of lines in the file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static List<String> readLinesAsList(Reader reader) throws IOException{\n\t\t// Make a BufferedReader out of the given Reader\n\t\tBufferedReader buffReader = new BufferedReader(reader);\n\t\t// Create a List of Strings to store the lines\n\t\tList<String> lines = new ArrayList<>();\n\t\t\n\t\t// Read the first line of the file\n\t\tString line = buffReader.readLine();\n\t\t\n\t\t// Continue to read lines until there are no more\n\t\twhile(line != null){\n\t\t\t// Add the line to the list\n\t\t\tlines.add(line);\n\t\t\t// Grab the next line\n\t\t\tline = buffReader.readLine();\n\t\t}\n\t\treturn lines;\n\t}\n\t\n\t/**\n\t * Reads the file at the given filepath into a byte array.\n\t * \n\t * @param filepath The path of the file to be read\n\t * @return The byte array of the given file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static byte[] readAsBytes(String filepath) throws IOException{\n\t\treturn Files.readAllBytes(Paths.get(filepath));\n\t}\n\t\n\t/**\n\t * Reads the given file into a byte array.\n\t * \n\t * @param file The {@link File} to be read\n\t * @return The byte array of the given file\n\t * @throws IOException If something goes wrong in reading the file\n\t */\n\tpublic static byte[] readAsBytes(File file) throws IOException{\n\t\treturn Files.readAllBytes(file.toPath());\n\t}\n\t\n\t/**\n\t * Writes the given string to the file given by the filepath.\n\t * Will create the file and its directories if they don't exist.\n\t * \n\t * @param filepath The path to save the file to\n\t * @param content The content of the file to be written\n\t * @throws IOException If something goes wrong in writing the file\n\t */\n\tpublic static void writeFile(String filepath, String content) throws IOException{\n\t\t// Create the File\n\t\tFile file = createFile(filepath);\n\t\t\n\t\t// Actually write to the file using a FileWriter\n\t\twriteFile(new FileWriter(file), content);\n\t}\n\t\n\t/**\n\t * Writes the given string to the file given in the {@link Writer}.\n\t * \n\t * @param writer The Writer to use in writing\n\t * @param content The content of the file to be written\n\t * @throws IOException If something goes wrong in writing the file\n\t */\n\tpublic static void writeFile(Writer writer, String content) throws IOException{\n\t\t// Make a BufferedWriter out of the given Writer\n\t\tBufferedWriter buffWriter = new BufferedWriter(writer);\n\t\t\n\t\t// Write the content to the file\n\t\tbuffWriter.write(content);\n\t\t\n\t\t// flush and close the writer\n\t\tbuffWriter.flush();\n\t\tbuffWriter.close();\n\t}\n\t\n\t/**\n\t * Writes the given lines to the file given in the {@link Writer}.\n\t * \n\t * @param writer The Writer to use in writing\n\t * @param lines The content of the file to be written\n\t * @throws IOException If something goes wrong in writing the file\n\t */\n\tpublic static void writeFile(Writer writer, Collection<String> lines) throws IOException{\n\t\twriteFile(writer, StringUtil.buildStringWithNewLines(lines));\n\t}\n\t\n\t/**\n\t * Writes the given byte array to the file path given\n\t * \n\t * @param filePath The path to write the file at\n\t * @param bytes The byte array contents of the file\n\t * @throws IOException If something goes wrong in writing the file\n\t */\n\tpublic static void writeFile(String filePath, byte[] bytes) throws IOException{\n\t\t// Create the File\n\t\tFile file = createFile(filePath);\n\t\t\n\t\t// Actually write to the file using a FileOutputStream\n\t\tFileOutputStream fos = new FileOutputStream(file);\n\t\tfos.write(bytes);\n\t\tfos.close();\n\t}\n\t\n\t/**\n\t * Creates a zip file using the file or directory at the given path.\n\t * \n\t * @param pathToZip The path to the file or directory to be zipped\n\t * @param zipPath The path (and name) of the zip file to be created\n\t * @throws IOException If something goes wrong in zipping the files\n\t */\n\tpublic static void zipFile(String pathToZip, String zipPath) throws IOException{\n\t\tzipFile(new File(pathToZip), zipPath);\n\t}\n\t\n\t/**\n\t * Creates a zip file using the given File (can be a file or directory).\n\t * \n\t * @param fileToZip The File to be zipped (can be file or directory)\n\t * @param zipPath The path (and name) of the zip file to be created\n\t * @throws IOException If something goes wrong in zipping the file\n\t */\n\tpublic static void zipFile(File fileToZip, String zipPath) throws IOException{\n\t\t// Setup\n\t\tFileOutputStream fos = new FileOutputStream(zipPath);\n\t\tZipOutputStream zos = new ZipOutputStream(fos);\n\t\t\n\t\t// Call helper method (which handles if it's a directory)\n\t\tzipFile(fileToZip, fileToZip.getName(), zos);\n\t\tzos.close();\n\t\tfos.close();\n\t}\n\t\n\t/**\n\t * Creates a zip file using the given File (can be a file or directory), fileName for the file to be zipped,\n\t * and a ZipOutputStream to be used. Each call to this method creates a single entry in the zip file, and\n\t * if it's a directory, it will make recursive calls for all children of the directory.\n\t * \n\t * @param fileToZip The File to be zipped (can be file or directory)\n\t * @param fileName The name of the file to be zipped\n\t * @param zos The ZipOutputStream to be used in zipping\n\t * @throws IOException If something goes wrong in zipping the file\n\t */\n\tprivate static void zipFile(File fileToZip, String fileName, ZipOutputStream zos) throws IOException{\n\t\tif(fileToZip.isHidden()){\n\t\t\treturn;\n\t\t}\n\t\t\n\t\t// Handle directories differently\n\t\tif(fileToZip.isDirectory()){\n\t\t\t// If we're missing the ending /, add it\n\t\t\tif(!fileName.endsWith(\"/\")){\n\t\t\t\tfileName += \"/\";\n\t\t\t}\n\t\t\t\n\t\t\t// Add the directory as a zip entry\n\t\t\tzos.putNextEntry(new ZipEntry(fileName));\n\t\t\tzos.closeEntry();\n\t\t\t\n\t\t\t// Grab the children of the directory\n\t\t\tFile[] children = fileToZip.listFiles();\n\t\t\tif(children == null){\n\t\t\t\treturn;\n\t\t\t}\n\t\t\t\n\t\t\t// Add all the children to the zip\n\t\t\tfor(File childFile: children){\n\t\t\t\tzipFile(childFile, fileName + \"/\" + childFile.getName(), zos);\n\t\t\t}\n\t\t\treturn;\n\t\t}\n\t\t\n\t\t// If it's not a directory, just zip the single file\n\t\tFileInputStream fis = new FileInputStream(fileToZip);\n\t\tZipEntry zipEntry = new ZipEntry(fileName);\n\t\tzos.putNextEntry(zipEntry);\n\t\tbyte[] bytes = new byte[1024];\n\t\tint length;\n\t\twhile((length = fis.read(bytes)) >= 0){\n\t\t\tzos.write(bytes, 0, length);\n\t\t}\n\t\tfis.close();\n\t}\n\t\n\t/**\n\t * Unzips a zip file into the given destination path.\n\t * \n\t * @param zipPath The path to the zip file\n\t * @param destinationPath The path to extract the zip file contents to\n\t * @throws IOException If something goes wrong in unzipping the file\n\t */\n\tpublic static void unzipFile(String zipPath, String destinationPath) throws IOException{\n\t\tunzipFile(zipPath, new File(destinationPath));\n\t}\n\t\n\t/**\n\t * Unzips a zip file into the given destination directory.\n\t * \n\t * @param zipPath The path to the zip file\n\t * @param destDirectory The directory to extract the zip file contents to\n\t * @throws IOException If something goes wrong in unzipping the file\n\t */\n\tpublic static void unzipFile(String zipPath, File destDirectory) throws IOException{\n\t\t// Setup\n\t\tbyte[] buffer = new byte[1024];\n\t\tZipInputStream zis = new ZipInputStream(new FileInputStream(zipPath));\n\t\tZipEntry zipEntry = zis.getNextEntry();\n\t\t\n\t\t// Extract each file\n\t\twhile(zipEntry != null){\n\t\t\tFile newFile = new File(destDirectory, zipEntry.getName());\n\t\t\t\n\t\t\t// Check destination path to prevent zip slip\n\t\t\tString destDirPath = destDirectory.getCanonicalPath();\n\t\t\tString destFilePath = newFile.getCanonicalPath();\n\t\t\tif(!destFilePath.startsWith(destDirPath + File.separator)){\n\t\t\t\tthrow new IOException(\"Entry is outside of the target dir: \" + zipEntry.getName());\n\t\t\t}\n\t\t\t\n\t\t\t// If it's a directory, make the directory instead of a file\n\t\t\tif(zipEntry.isDirectory()){\n\t\t\t\tif(!newFile.mkdirs() && !newFile.exists()){\n\t\t\t\t\tthrow new IOException(\"Failed to create new directory at \" + newFile.getAbsolutePath());\n\t\t\t\t}\n\t\t\t}else{\n\t\t\t\t// Write the new file\n\t\t\t\tFileOutputStream fos = new FileOutputStream(newFile);\n\t\t\t\tint length;\n\t\t\t\twhile((length = zis.read(buffer)) > 0){\n\t\t\t\t\tfos.write(buffer, 0, length);\n\t\t\t\t}\n\t\t\t\t\n\t\t\t\t// Close the new file and prepare for next one\n\t\t\t\tfos.close();\n\t\t\t}\n\t\t\tzipEntry = zis.getNextEntry();\n\t\t}\n\t\tzis.closeEntry();\n\t\tzis.close();\n\t}\n}\n", EditableJavaClass.builder().packageName("com.github.tadukoo.util").importName("java.io.BufferedReader", false).importName("java.io.BufferedWriter", false).importName("java.io.File", false).importName("java.io.FileInputStream", false).importName("java.io.FileNotFoundException", false).importName("java.io.FileOutputStream", false).importName("java.io.FileReader", false).importName("java.io.FileWriter", false).importName("java.io.IOException", false).importName("java.io.Reader", false).importName("java.io.Writer", false).importName("java.nio.file.Files", false).importName("java.nio.file.Path", false).importName("java.nio.file.Paths", false).importName("java.util.ArrayList", false).importName("java.util.Collection", false).importName("java.util.Comparator", false).importName("java.util.List", false).importName("java.util.stream.Collectors", false).importName("java.util.stream.Stream", false).importName("java.util.zip.ZipEntry", false).importName("java.util.zip.ZipInputStream", false).importName("java.util.zip.ZipOutputStream", false).javadoc(EditableJavadoc.builder().content("Util functions for dealing with Files.").author("Logan Ferree (Tadukoo)").version("Beta v.0.5.2").since("Pre-Alpha").build()).visibility(Visibility.PUBLIC).isFinal().className("FileUtil").method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().condensed().content("Not allowed to create a FileUtil").build()).visibility(Visibility.PRIVATE).returnType("FileUtil").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Finds the file extension for the given filepath, if one exists.").param("filepath", "The filepath to grab the extension for").returnVal("The found file extension, or null").build()).visibility(Visibility.PUBLIC).isStatic().returnType("String").name("getFileExtension").parameter("String filepath").line("// Find the last dot in the filepath").line("int lastDotIndex = filepath.lastIndexOf('.');").line("").line("// If one isn't found, return null").line("if(lastDotIndex == -1){").line("\treturn null;").line("}else{").line("\t// If we found a dot, return whatever's after it").line("\treturn filepath.substring(lastDotIndex + 1);").line("}").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of all Files in the given directory and any of").content("its subdirectories.").param("directoryPath", "The path to the directory to check").returnVal("A List of all Files in the directory and its subdirectories").throwsInfo("IOException", "If something goes wrong in listing the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<File>").name("listAllFiles").parameter("Path directoryPath").throwType("IOException").line("try(Stream<Path> pathStream = Files.walk(directoryPath)){").line("\treturn pathStream").line("\t\t\t.filter(Files::isRegularFile)").line("\t\t\t.map(Path::toFile)").line("\t\t\t.collect(Collectors.toList());").line("}").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of all Files in the given directory and any of its").content("subdirectories.").param("directory", "The directory (as a File) to check").returnVal("A List of all Files in the directory and its subdirectories").throwsInfo("IOException", "If something goes wrong in listing the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<File>").name("listAllFiles").parameter("File directory").throwType("IOException").line("return listAllFiles(directory.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of all Files in the given directory and any of its").content("subdirectories.").param("directory", "The directory path (as a String) to check").returnVal("A List of all Files in the directory and its subdirectories").throwsInfo("IOException", "If something goes wrong in listing the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<File>").name("listAllFiles").parameter("String directory").throwType("IOException").line("return listAllFiles(Paths.get(directory));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of all files in the given directory and any of").content("its subdirectories. The files are returned as Paths.").param("directory", "The path to the directory to check").returnVal("A List of all files in the directory and its subdirectories as Paths").throwsInfo("IOException", "If something goes wrong in listing the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<Path>").name("listAllFilesAsPaths").parameter("Path directory").throwType("IOException").line("try(Stream<Path> pathStream = Files.walk(directory)){").line("\treturn pathStream").line("\t\t\t.filter(Files::isRegularFile)").line("\t\t\t.collect(Collectors.toList());").line("}").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of all files in the given directory and any of its").content("subdirectories. The files are returned as Paths.").param("directory", "The directory (as a File) to check").returnVal("A List of all files in the directory and its subdirectories as Paths").throwsInfo("IOException", "If something goes wrong in listing the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<Path>").name("listAllFilesAsPaths").parameter("File directory").throwType("IOException").line("return listAllFilesAsPaths(directory.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of all files in the given directory and any of its").content("subdirectories. The files are returned as Paths.").param("directory", "The directory path (as a String) to check").returnVal("A List of all files in the directory and its subdirectories as Paths").throwsInfo("IOException", "If something goes wrong in listing the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<Path>").name("listAllFilesAsPaths").parameter("String directory").throwType("IOException").line("return listAllFilesAsPaths(Paths.get(directory));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a file at the given filepath, including any directories necessary,").content("and returns the {@link File} object to be used.").param("filepath", "The path for the File to be created").returnVal("The newly created File").throwsInfo("IOException", "If something goes wrong in creating the file").build()).annotation(EditableJavaAnnotation.builder().name("SuppressWarnings").parameter("value", "\"ResultOfMethodCallIgnored\"").build()).visibility(Visibility.PUBLIC).isStatic().returnType("File").name("createFile").parameter("String filepath").throwType("IOException").line("// Create a File object from the given filepath").line("File file = new File(filepath);").line("").line("// If a directory is specified, create it if it doesn't exist").line("File parentFile = file.getParentFile();").line("if(parentFile != null && !parentFile.exists()){").line("\tfile.getParentFile().mkdirs();").line("}").line("").line("// If the file doesn't exist, create it").line("if(!file.exists()){").line("\tfile.createNewFile();").line("}").line("").line("return file;").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Deletes the file at the given filepath").param("filepath", "The path for the File to be deleted").returnVal("If the file was deleted or not").build()).visibility(Visibility.PUBLIC).isStatic().returnType("boolean").name("deleteFile").parameter("String filepath").line("// Create the File object from the given filepath").line("File file = new File(filepath);").line("").line("return file.delete();").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a directory at the given directoryPath, including any parent directories").content("necessary, and returns the {@link File} object to be used.").param("directoryPath", "The path for the directory to be created").returnVal("The newly created directory").build()).visibility(Visibility.PUBLIC).isStatic().returnType("File").name("createDirectory").parameter("String directoryPath").line("// Create a File object from the given filepath").line("File directory = new File(directoryPath);").line("").line("// Make the directory and its parent folders").line("if(!directory.mkdirs() && !directory.exists()){").line("\tthrow new IllegalStateException(\"Failed to create folder: \" + directoryPath + \"!\");").line("}").line("").line("return directory;").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Deletes the directory at the given directoryPath, including any files contained").content("within it.").param("directoryPath", "The path to the directory to be deleted").throwsInfo("IOException", "If anything goes wrong").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("deleteDirectory").parameter("String directoryPath").throwType("IOException").line("try(Stream<Path> pathStream = Files.walk(Path.of(directoryPath))){").line("\t//noinspection ResultOfMethodCallIgnored").line("\tpathStream").line("\t\t\t.sorted(Comparator.reverseOrder())").line("\t\t\t.map(Path::toFile)").line("\t\t\t.forEach(File::delete);").line("}").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a new {@link BufferedReader} for the file at the given filepath.").param("filepath", "The path of the file to be read").returnVal("A {@link BufferedReader} for the given file").throwsInfo("FileNotFoundException", "If the file can't be found").build()).visibility(Visibility.PUBLIC).isStatic().returnType("BufferedReader").name("setupFileReader").parameter("String filepath").throwType("FileNotFoundException").line("return new BufferedReader(new FileReader(filepath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a new {@link BufferedReader} for the given {@link File}.").param("file", "The {@link File} to be read").returnVal("A {@link BufferedReader} for the given {@link File}").throwsInfo("FileNotFoundException", "If the file can't be found").build()).visibility(Visibility.PUBLIC).isStatic().returnType("BufferedReader").name("setupFileReader").parameter("File file").throwType("FileNotFoundException").line("return new BufferedReader(new FileReader(file));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Checks if the file at the given filepath exists.").param("filepath", "The path of the file to be checked").returnVal("true if the file exists, false if it doesn't").build()).visibility(Visibility.PUBLIC).isStatic().returnType("boolean").name("exists").parameter("String filepath").line("return Files.exists(Paths.get(filepath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Checks if the given {@link File} exists.").param("file", "The {@link File} to be checked").returnVal("true if the file exists, false if it doesn't").build()).visibility(Visibility.PUBLIC).isStatic().returnType("boolean").name("exists").parameter("File file").line("return Files.exists(file.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Checks if the file at the given filepath does not exist.").param("filepath", "The path of the file to be checked").returnVal("false if the file exists, true if it doesn't").build()).visibility(Visibility.PUBLIC).isStatic().returnType("boolean").name("notExists").parameter("String filepath").line("return Files.notExists(Paths.get(filepath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Checks if the given {@link File} does not exist.").param("file", "The {@link File} to be checked").returnVal("false if the file exists, true if it doesn't").build()).visibility(Visibility.PUBLIC).isStatic().returnType("boolean").name("notExists").parameter("File file").line("return Files.notExists(file.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Reads the file at the given filepath as a String").param("filepath", "The path of the file to be read").returnVal("A String representing the contents of the file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("String").name("readAsString").parameter("String filepath").throwType("IOException").line("return Files.readString(Paths.get(filepath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Reads the given {@link File} as a String").param("file", "The {@link File} to be read").returnVal("A String representing the contents of the file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("String").name("readAsString").parameter("File file").throwType("IOException").line("return Files.readString(file.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of Strings for each line in the file being read from the given filepath.").param("filepath", "The path to the file to be read").returnVal("A List of lines in the file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<String>").name("readLinesAsList").parameter("String filepath").throwType("IOException").line("return Files.readAllLines(Paths.get(filepath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of Strings for each line in the file being read.").param("file", "The {@link File} to read").returnVal("A List of lines in the file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<String>").name("readLinesAsList").parameter("File file").throwType("IOException").line("return Files.readAllLines(file.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a List of Strings for each line in the file being read in the").content("given {@link Reader}.").param("reader", "The Reader to use in reading").returnVal("A List of lines in the file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("List<String>").name("readLinesAsList").parameter("Reader reader").throwType("IOException").line("// Make a BufferedReader out of the given Reader").line("BufferedReader buffReader = new BufferedReader(reader);").line("// Create a List of Strings to store the lines").line("List<String> lines = new ArrayList<>();").line("").line("// Read the first line of the file").line("String line = buffReader.readLine();").line("").line("// Continue to read lines until there are no more").line("while(line != null){").line("\t// Add the line to the list").line("\tlines.add(line);").line("\t// Grab the next line").line("\tline = buffReader.readLine();").line("}").line("return lines;").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Reads the file at the given filepath into a byte array.").param("filepath", "The path of the file to be read").returnVal("The byte array of the given file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("byte[]").name("readAsBytes").parameter("String filepath").throwType("IOException").line("return Files.readAllBytes(Paths.get(filepath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Reads the given file into a byte array.").param("file", "The {@link File} to be read").returnVal("The byte array of the given file").throwsInfo("IOException", "If something goes wrong in reading the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("byte[]").name("readAsBytes").parameter("File file").throwType("IOException").line("return Files.readAllBytes(file.toPath());").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Writes the given string to the file given by the filepath.").content("Will create the file and its directories if they don't exist.").param("filepath", "The path to save the file to").param("content", "The content of the file to be written").throwsInfo("IOException", "If something goes wrong in writing the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("writeFile").parameter("String filepath").parameter("String content").throwType("IOException").line("// Create the File").line("File file = createFile(filepath);").line("").line("// Actually write to the file using a FileWriter").line("writeFile(new FileWriter(file), content);").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Writes the given string to the file given in the {@link Writer}.").param("writer", "The Writer to use in writing").param("content", "The content of the file to be written").throwsInfo("IOException", "If something goes wrong in writing the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("writeFile").parameter("Writer writer").parameter("String content").throwType("IOException").line("// Make a BufferedWriter out of the given Writer").line("BufferedWriter buffWriter = new BufferedWriter(writer);").line("").line("// Write the content to the file").line("buffWriter.write(content);").line("").line("// flush and close the writer").line("buffWriter.flush();").line("buffWriter.close();").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Writes the given lines to the file given in the {@link Writer}.").param("writer", "The Writer to use in writing").param("lines", "The content of the file to be written").throwsInfo("IOException", "If something goes wrong in writing the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("writeFile").parameter("Writer writer").parameter("Collection<String> lines").throwType("IOException").line("writeFile(writer, StringUtil.buildStringWithNewLines(lines));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Writes the given byte array to the file path given").param("filePath", "The path to write the file at").param("bytes", "The byte array contents of the file").throwsInfo("IOException", "If something goes wrong in writing the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("writeFile").parameter("String filePath").parameter("byte[] bytes").throwType("IOException").line("// Create the File").line("File file = createFile(filePath);").line("").line("// Actually write to the file using a FileOutputStream").line("FileOutputStream fos = new FileOutputStream(file);").line("fos.write(bytes);").line("fos.close();").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a zip file using the file or directory at the given path.").param("pathToZip", "The path to the file or directory to be zipped").param("zipPath", "The path (and name) of the zip file to be created").throwsInfo("IOException", "If something goes wrong in zipping the files").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("zipFile").parameter("String pathToZip").parameter("String zipPath").throwType("IOException").line("zipFile(new File(pathToZip), zipPath);").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a zip file using the given File (can be a file or directory).").param("fileToZip", "The File to be zipped (can be file or directory)").param("zipPath", "The path (and name) of the zip file to be created").throwsInfo("IOException", "If something goes wrong in zipping the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("zipFile").parameter("File fileToZip").parameter("String zipPath").throwType("IOException").line("// Setup").line("FileOutputStream fos = new FileOutputStream(zipPath);").line("ZipOutputStream zos = new ZipOutputStream(fos);").line("").line("// Call helper method (which handles if it's a directory)").line("zipFile(fileToZip, fileToZip.getName(), zos);").line("zos.close();").line("fos.close();").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Creates a zip file using the given File (can be a file or directory), fileName for the file to be zipped,").content("and a ZipOutputStream to be used. Each call to this method creates a single entry in the zip file, and").content("if it's a directory, it will make recursive calls for all children of the directory.").param("fileToZip", "The File to be zipped (can be file or directory)").param("fileName", "The name of the file to be zipped").param("zos", "The ZipOutputStream to be used in zipping").throwsInfo("IOException", "If something goes wrong in zipping the file").build()).visibility(Visibility.PRIVATE).isStatic().returnType("void").name("zipFile").parameter("File fileToZip").parameter("String fileName").parameter("ZipOutputStream zos").throwType("IOException").line("if(fileToZip.isHidden()){").line("\treturn;").line("}").line("").line("// Handle directories differently").line("if(fileToZip.isDirectory()){").line("\t// If we're missing the ending /, add it").line("\tif(!fileName.endsWith(\"/\")){").line("\t\tfileName += \"/\";").line("\t}").line("\t").line("\t// Add the directory as a zip entry").line("\tzos.putNextEntry(new ZipEntry(fileName));").line("\tzos.closeEntry();").line("\t").line("\t// Grab the children of the directory").line("\tFile[] children = fileToZip.listFiles();").line("\tif(children == null){").line("\t\treturn;").line("\t}").line("\t").line("\t// Add all the children to the zip").line("\tfor(File childFile: children){").line("\t\tzipFile(childFile, fileName + \"/\" + childFile.getName(), zos);").line("\t}").line("\treturn;").line("}").line("").line("// If it's not a directory, just zip the single file").line("FileInputStream fis = new FileInputStream(fileToZip);").line("ZipEntry zipEntry = new ZipEntry(fileName);").line("zos.putNextEntry(zipEntry);").line("byte[] bytes = new byte[1024];").line("int length;").line("while((length = fis.read(bytes)) >= 0){").line("\tzos.write(bytes, 0, length);").line("}").line("fis.close();").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Unzips a zip file into the given destination path.").param("zipPath", "The path to the zip file").param("destinationPath", "The path to extract the zip file contents to").throwsInfo("IOException", "If something goes wrong in unzipping the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("unzipFile").parameter("String zipPath").parameter("String destinationPath").throwType("IOException").line("unzipFile(zipPath, new File(destinationPath));").build()).method(EditableJavaMethod.builder().javadoc(EditableJavadoc.builder().content("Unzips a zip file into the given destination directory.").param("zipPath", "The path to the zip file").param("destDirectory", "The directory to extract the zip file contents to").throwsInfo("IOException", "If something goes wrong in unzipping the file").build()).visibility(Visibility.PUBLIC).isStatic().returnType("void").name("unzipFile").parameter("String zipPath").parameter("File destDirectory").throwType("IOException").line("// Setup").line("byte[] buffer = new byte[1024];").line("ZipInputStream zis = new ZipInputStream(new FileInputStream(zipPath));").line("ZipEntry zipEntry = zis.getNextEntry();").line("").line("// Extract each file").line("while(zipEntry != null){").line("\tFile newFile = new File(destDirectory, zipEntry.getName());").line("\t").line("\t// Check destination path to prevent zip slip").line("\tString destDirPath = destDirectory.getCanonicalPath();").line("\tString destFilePath = newFile.getCanonicalPath();").line("\tif(!destFilePath.startsWith(destDirPath + File.separator)){").line("\t\tthrow new IOException(\"Entry is outside of the target dir: \" + zipEntry.getName());").line("\t}").line("\t").line("\t// If it's a directory, make the directory instead of a file").line("\tif(zipEntry.isDirectory()){").line("\t\tif(!newFile.mkdirs() && !newFile.exists()){").line("\t\t\tthrow new IOException(\"Failed to create new directory at \" + newFile.getAbsolutePath());").line("\t\t}").line("\t}else{").line("\t\t// Write the new file").line("\t\tFileOutputStream fos = new FileOutputStream(newFile);").line("\t\tint length;").line("\t\twhile((length = zis.read(buffer)) > 0){").line("\t\t\tfos.write(buffer, 0, length);").line("\t\t}").line("\t\t").line("\t\t// Close the new file and prepare for next one").line("\t\tfos.close();").line("\t}").line("\tzipEntry = zis.getNextEntry();").line("}").line("zis.closeEntry();").line("zis.close();").build()).build());
    }
}

