package cdc.issues;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Objects;

import cdc.issues.locations.Location;
import cdc.issues.rules.RuleId;
import cdc.util.lang.Checks;

/**
 * Data issue identifier.
 * <p>
 * It is composed of:
 * <ul>
 * <li>a rule identifier (domain and rule name)
 * <li>rule parameters
 * <li>a project name
 * <li>locations
 * </ul>
 * <b>Note:</b> several occurrences of issues may have the same identifier.
 * They will differ by their snapshot.
 *
 *
 * @author Damien Carbonne
 */
public class IssueId implements Comparable<IssueId> {
    private final RuleId ruleId;
    private final Params params;
    private final String project;
    private final Location[] locations;

    @Deprecated(forRemoval = true)
    public IssueId(RuleId ruleId,
                   Params params,
                   String project,
                   Location... locations) {
        this.ruleId = Checks.isNotNull(ruleId, "ruleId");
        this.params = Checks.isNotNull(params, "params");
        this.project = project;
        this.locations = Checks.isNotNull(locations, "locations").clone();
    }

    @Deprecated(forRemoval = true)
    public IssueId(String domain,
                   String name,
                   Params params,
                   String project,
                   Location... locations) {
        this(new RuleId(domain, name),
             params,
             project,
             locations);
    }

    @Deprecated(forRemoval = true)
    public IssueId(RuleId ruleId,
                   Params params,
                   Location... locations) {
        this(ruleId,
             params,
             null,
             locations);
    }

    @Deprecated(forRemoval = true)
    public IssueId(String domain,
                   String name,
                   Params params,
                   Location... locations) {
        this(domain,
             name,
             params,
             null,
             locations);
    }

    @Deprecated(forRemoval = true)
    public IssueId(String domain,
                   Enum<?> name,
                   Params params,
                   String project,
                   Location... locations) {
        this(domain,
             name.name(),
             params,
             project,
             locations);
    }

    @Deprecated(forRemoval = true)
    public IssueId(String domain,
                   Enum<?> name,
                   Params params,
                   Location... locations) {
        this(domain,
             name,
             params,
             null,
             locations);
    }

    public RuleId getRuleId() {
        return ruleId;
    }

    public String getDomain() {
        return ruleId.getDomain();
    }

    public String getName() {
        return ruleId.getName();
    }

    public <T extends Enum<T>> T getName(Class<T> cls) {
        return ruleId.getName(cls);
    }

    public Params getParams() {
        return params;
    }

    public String getProject() {
        return project;
    }

    public Location[] getLocations() {
        return locations.clone();
    }

    @Override
    public int hashCode() {
        return Objects.hash(ruleId,
                            params,
                            project)
                + 31 * Arrays.hashCode(locations);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof IssueId)) {
            return false;
        }
        final IssueId other = (IssueId) object;
        return this.ruleId.equals(other.ruleId)
                && Objects.equals(params, other.params)
                && Objects.equals(project, other.project)
                && Arrays.equals(this.locations, other.locations);
    }

    private static final Comparator<String> STRING_COMPARATOR =
            Comparator.nullsFirst(Comparator.naturalOrder());

    @Override
    public int compareTo(IssueId other) {
        final int ruleIdCmp = ruleId.compareTo(other.ruleId);
        if (ruleIdCmp != 0) {
            return ruleIdCmp;
        }
        final int paramsCmp = params.compareTo(other.params);
        if (paramsCmp != 0) {
            return paramsCmp;
        }
        final int projectCmp = STRING_COMPARATOR.compare(project, other.project);
        if (projectCmp != 0) {
            return projectCmp;
        }
        return Arrays.compare(locations, other.locations);
    }

    @Override
    public String toString() {
        return "[" + ruleId + " " + params + " " + project + " " + Arrays.toString(locations) + "]";
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private String domain;
        private String name;
        private Params params = Params.NO_PARAMS;
        private String project;
        private Location[] locations = {};

        private Builder() {
        }

        public Builder ruleId(RuleId ruleId) {
            this.domain = ruleId.getDomain();
            this.name = ruleId.getName();
            return this;
        }

        public Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder name(Enum<?> name) {
            this.name = name.name();
            return this;
        }

        public Builder params(Params params) {
            this.params = params;
            return this;
        }

        public Builder project(String project) {
            this.project = project;
            return this;
        }

        public Builder locations(Location... locations) {
            Checks.isNotNull(locations, "locations");

            this.locations = locations;
            return this;
        }

        public IssueId build() {
            return new IssueId(domain,
                               name,
                               params,
                               project,
                               locations);
        }
    }
}