package cdc.issues;

import java.util.Objects;

import cdc.util.lang.Checks;

/**
 * Definition of an effective parameter.
 * <p>
 * It is a (name, value) pair, where name can not be {@code null}.
 *
 * @author Damien Carbonne
 */
public final class Param {
    private static final String NAME = "name";

    private final String name;
    private final String value;

    private Param(Builder builder) {
        this.name = Checks.isNotNull(builder.name, NAME);
        this.value = builder.value;
    }

    /**
     * @return The parameter name.
     */
    public String getName() {
        return name;
    }

    /**
     * @return The parameter value.
     */
    public String getValue() {
        return value;
    }

    @Override
    public int hashCode() {
        return Objects.hash(name,
                            value);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof Param)) {
            return false;
        }
        final Param other = (Param) object;
        return Objects.equals(name, other.name)
                && Objects.equals(value, other.value);
    }

    @Override
    public String toString() {
        return "[" + name + "=" + value + "]";
    }

    /**
     * @return A new {@link Builder} of {@link Param}.
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder of {@link Param}.
     */
    public static final class Builder {
        private String name;
        private String value;

        private Builder() {
        }

        public Builder name(String name) {
            this.name = Checks.isNotNull(name, NAME);
            return this;
        }

        public Builder value(String value) {
            this.value = value;
            return this;
        }

        public Param build() {
            return new Param(this);
        }
    }
}