package cdc.issues;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import cdc.util.lang.Checks;

/**
 * Set of effective parameters.
 * <p>
 * It is a set of (name, value) pairs.
 *
 * @author Damien Carbonne
 */
public final class Params implements Comparable<Params> {
    private static final String NAME = "name";

    public static final Params NO_PARAMS = builder().build();

    private final Map<String, String> map = new HashMap<>();

    private Params(Builder builder) {
        for (final Map.Entry<String, String> entry : builder.map.entrySet()) {
            this.map.put(entry.getKey(), entry.getValue());
        }
    }

    public boolean isEmpty() {
        return map.isEmpty();
    }

    /**
     * @return The parameters names.
     */
    public Set<String> getNames() {
        return map.keySet();
    }

    /**
     * @return The sorted parameters names.
     */
    public List<String> getSortedNames() {
        return map.keySet()
                  .stream()
                  .sorted()
                  .collect(Collectors.toList());
    }

    /**
     * Returns the value associated to a parameter, or {@code null}.
     *
     * @param name The parameter name.
     * @return The value of parameter named {@code name}.
     */
    public String getValue(String name) {
        return map.get(name);
    }

    /**
     * Returns the value associated to a parameter, or a default value.
     *
     * @param name The parameter name.
     * @param def The default value.
     * @return The value of parameter named {@code name}, or {@code def}.
     */
    public String getValue(String name,
                           String def) {
        return map.getOrDefault(name, def);
    }

    @Override
    public int hashCode() {
        return map.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof Params)) {
            return false;
        }
        final Params other = (Params) object;
        return this.map.equals(other.map);
    }

    @Override
    public int compareTo(Params other) {
        return toString().compareTo(other.toString());
    }

    @Override
    public String toString() {
        return getSortedNames().stream()
                               .map(name -> name + ":" + getValue(name))
                               .collect(Collectors.joining(" ", "[", "]"));
    }

    /**
     * @return A new {@link Builder} of {@link Params}.
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder of {@link Params}.
     */
    public static final class Builder {
        private final Map<String, String> map = new HashMap<>();

        private Builder() {
        }

        public Builder param(String name,
                             String value) {
            Checks.isNotNull(name, NAME);

            map.put(name, value);
            return this;
        }

        public Builder params(Collection<Param> params) {
            for (final Param param : params) {
                map.put(param.getName(), param.getValue());
            }
            return this;
        }

        public Params build() {
            return new Params(this);
        }
    }
}