package cdc.issues.rules;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import cdc.util.lang.Checks;

public abstract class AbstractIssueDetector<T> implements IssuesDetector<T> {
    protected final Factory<T> factory;
    protected final String project;
    protected final String snapshot;
    protected final Set<ConfiguredRule> enabledConfiguredRules;
    protected final Set<Rule> enabledRules;

    protected AbstractIssueDetector(Factory<T> factory,
                                    String project,
                                    String snapshot,
                                    Set<ConfiguredRule> configuredRules) {
        Checks.isNotNullOrEmpty(configuredRules, "configuredRules");
        this.factory = Checks.isNotNull(factory, "factory");
        this.project = project;
        this.snapshot = snapshot;
        this.enabledConfiguredRules = Collections.unmodifiableSet(new HashSet<>(configuredRules));
        final Set<Rule> rules = new HashSet<>();
        for (final ConfiguredRule crule : configuredRules) {
            if (!factory.getSupportedRules().contains(crule.getRule())) {
                throw new IllegalArgumentException("Non supported rule");
            }
            rules.add(crule.getRule());
        }
        this.enabledRules = Collections.unmodifiableSet(rules);
    }

    @Override
    public Factory<T> getFactory() {
        return factory;
    }

    @Override
    public final String getProject() {
        return project;
    }

    @Override
    public final String getSnapshot() {
        return snapshot;
    }

    @Override
    public final Set<Rule> getEnabledRules() {
        return enabledRules;
    }

    @Override
    public final Set<ConfiguredRule> getEnabledConfiguredRules() {
        return enabledConfiguredRules;
    }

    @Override
    public String toString() {
        return IssuesDetector.toString(this);
    }
}