package cdc.issues.api;

import java.util.Arrays;
import java.util.Objects;

import cdc.util.lang.Checks;

/**
 * Issue identifier.
 * <p>
 * It is composed of issue type and locations.
 *
 * @author Damien Carbonne
 */
public class IssueId {
    private final RuleId ruleId;
    private final IssueParams params;
    private final IssueLocation[] locations;

    public IssueId(RuleId ruleId,
                   IssueParams params,
                   IssueLocation[] locations) {
        this.ruleId = Checks.isNotNull(ruleId, "ruleId");
        this.params = Checks.isNotNull(params, "params");
        this.locations = Checks.isNotNull(locations, "locations").clone();
        if (locations.length == 0) {
            throw new IllegalArgumentException("Empty locations");
        }
    }

    public IssueId(String domain,
                   String name,
                   IssueParams params,
                   IssueLocation[] locations) {
        this(new RuleId(domain, name),
             params,
             locations);
    }

    public IssueId(RuleId ruleId,
                   IssueLocation[] locations) {
        this(ruleId,
             IssueParams.NO_PARAMS,
             locations);
    }

    public IssueId(String domain,
                   String name,
                   IssueLocation[] locations) {
        this(domain,
             name,
             IssueParams.NO_PARAMS,
             locations);
    }

    public IssueId(String domain,
                   Enum<?> name,
                   IssueParams params,
                   IssueLocation[] locations) {
        this(domain,
             name.name(),
             params,
             locations);
    }

    public IssueId(String domain,
                   Enum<?> name,
                   IssueLocation[] locations) {
        this(domain,
             name,
             IssueParams.NO_PARAMS,
             locations);
    }

    public RuleId getRuleId() {
        return ruleId;
    }

    public String getDomain() {
        return ruleId.getDomain();
    }

    public String getName() {
        return ruleId.getName();
    }

    public <T extends Enum<T>> T getName(Class<T> cls) {
        return ruleId.getName(cls);
    }

    public IssueParams getParams() {
        return params;
    }

    public IssueLocation[] getLocations() {
        return locations.clone();
    }

    @Override
    public int hashCode() {
        return Objects.hash(ruleId,
                            params)
                + 31 * Arrays.hashCode(locations);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof IssueId)) {
            return false;
        }
        final IssueId other = (IssueId) object;
        return this.ruleId.equals(other.ruleId)
                && this.params.equals(other.params)
                && Arrays.equals(this.locations, other.locations);
    }

    @Override
    public String toString() {
        return "[" + ruleId + " " + params + " " + Arrays.toString(locations) + "]";
    }
}