package cdc.issues.api;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * A collector of issues that can delegate issues.
 *
 * @author Damien Carbonne
 */
public class IssuesCollector<I extends Issue> implements IssuesHandler<I> {
    private final List<I> issues = new ArrayList<>();
    private final IssuesHandler<I> delegate;

    public IssuesCollector(IssuesHandler<I> delegate) {
        this.delegate = delegate;
    }

    public IssuesCollector() {
        this.delegate = null;
    }

    /**
     * @return The delegate (possibly {@code null}).
     */
    public IssuesHandler<I> getDelegate() {
        return delegate;
    }

    /**
     * Clears collected issues.
     */
    public void clear() {
        issues.clear();
    }

    /**
     * @return The list of collected issues.
     */
    public List<I> getIssues() {
        return issues;
    }

    /**
     * Returns a list of issues having a severity.
     *
     * @param severity The severity.
     * @return A list of issues that have a certain {@code severity}.
     */
    public List<I> getIssues(IssueSeverity severity) {
        return issues.stream()
                     .filter(i -> i.getSeverity() == severity)
                     .collect(Collectors.toList());
    }

    @Override
    public void issue(I issue) {
        issues.add(issue);
        if (delegate != null) {
            delegate.issue(issue);
        }
    }
}