package cdc.issues.api;

import java.io.File;

import cdc.util.lang.Checks;
import cdc.util.lang.FailureReaction;
import cdc.util.lang.Introspection;

public class IssuesFactory {
    private final IssuesFactoryFeatures features;

    public IssuesFactory(IssuesFactoryFeatures features) {
        this.features = features;
    }

    public IssuesFactoryFeatures getFeatures() {
        return features;
    }

    public boolean canExportTo(File file) {
        if (file == null) {
            return false;
        } else {
            final IssuesFormat format = IssuesFormat.from(file);
            return format != null && format.isExportFormat();
        }
    }

    /**
     * Creates an IssuesWriter.
     * <p>
     * See {@link IssuesFormat#isExportFormat()} for a list of supported formats.
     *
     * @param file The file for which an IssuesWriter must be created.
     * @return A new IssueWriter which can produce {@code file}.
     * @throws IllegalArgumentException When {@code file} is {@code null} or has an unrecognized extension.
     */
    public IssuesWriter createIssuesWriter(File file) {
        Checks.isNotNull(file, "file");

        final IssuesFormat format = IssuesFormat.from(file);
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
                return createIssuesWriter("cdc.issues.core.WorkbookIssuesWriter");
            case JSON:
                return createIssuesWriter("cdc.issues.core.JsonIssuesWriter");
            case XML:
                return createIssuesWriter("cdc.issues.core.XmlIssuesWriter");
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesWriter for " + file);
    }

    private IssuesWriter createIssuesWriter(String className) {
        final Class<? extends IssuesWriter> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesWriter.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes = { IssuesFactoryFeatures.class };
        return Introspection.newInstance(cls, parameterTypes, FailureReaction.FAIL, getFeatures());
    }

    public boolean canImportFrom(File file) {
        if (file == null) {
            return false;
        } else {
            final IssuesFormat format = IssuesFormat.from(file);
            return format != null && format.isImportFormat();
        }
    }

    public IssuesReader createIssuesReader(File file) {
        Checks.isNotNull(file, "file");

        final IssuesFormat format = IssuesFormat.from(file);
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
                return createIssuesReader("cdc.issues.core.WorkbookIssuesReader");
            case JSON:
                return createIssuesReader("cdc.issues.core.JsonIssuesReader");
            case XML:
                return createIssuesReader("cdc.issues.core.XmlIssuesReader");
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesReader for " + file);
    }

    private IssuesReader createIssuesReader(String className) {
        final Class<? extends IssuesReader> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesReader.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes = { IssuesFactoryFeatures.class };
        return Introspection.newInstance(cls, parameterTypes, FailureReaction.FAIL, getFeatures());
    }
}