package cdc.issues.api;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import cdc.util.lang.Checks;

/**
 * Catalog of Rules.
 *
 * @author Damien Carbonne
 */
public class RulesCatalog {
    private final Map<RuleId, Rule> rules = new HashMap<>();
    private final Map<String, Set<Rule>> domains = new HashMap<>();

    public RulesCatalog() {
        super();
    }

    public void register(Rule rule) {
        Checks.isNotNull(rule, "rule");
        Checks.doesNotContainKey(rules, rule.getId(), "rules");

        rules.put(rule.getId(), rule);
        final Set<Rule> set = domains.computeIfAbsent(rule.getId().getDomain(), k -> new HashSet<>());
        set.add(rule);
    }

    /**
     * @return A set of domains for which rules are registered.
     */
    public Set<String> getDomains() {
        return domains.keySet();
    }

    /**
     * @return A set of known RuleIds.
     */
    public Set<RuleId> getRuleIds() {
        return rules.keySet();
    }

    /**
     * @return A set of known Rules.
     */
    public Collection<Rule> getRules() {
        return rules.values();
    }

    /**
     * @param domain The domain.
     * @return A set of rules associated to a domain and registered in this catalog.
     */
    public Set<Rule> getRules(String domain) {
        return domains.getOrDefault(domain, Collections.emptySet());
    }

    /**
     * @param id The RuleId.
     * @return The Rule associated to {@code id} or {@code null}.
     */
    public Rule getRuleOrNull(RuleId id) {
        return rules.get(id);
    }
}