package cdc.issues.api.locations;

import cdc.util.lang.Checks;

/**
 * Implementation of IssueLocation dedicated to text files.
 *
 * @author Damien Carbonne
 */
public class TextFileLocation extends AbstractLocation {
    private final String systemId;
    private final int lineNumber;
    private final int columnNumber;

    private TextFileLocation(String systemId,
                             int lineNumber,
                             int columnNumber) {
        this.systemId = Checks.isNotNull(systemId, "systemId");
        this.lineNumber = lineNumber;
        this.columnNumber = columnNumber;
        if (lineNumber < 1 && columnNumber >= 1) {
            throw new IllegalArgumentException("Non compliant numbers");
        }
    }

    @Override
    public String getTargetId() {
        return getSystemId();
    }

    @Override
    public String getPath() {
        if (lineNumber >= 1) {
            if (columnNumber >= 1) {
                return lineNumber + ":" + columnNumber;
            } else {
                return Integer.toString(lineNumber);
            }
        } else {
            return "";
        }
    }

    public String getSystemId() {
        return systemId;
    }

    public int getLineNumber() {
        return lineNumber;
    }

    public int getColumnNumber() {
        return columnNumber;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        return super.equals(object);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String systemId;
        private int lineNumber = 0;
        private int columnNumber = 0;

        private Builder() {
            super();
        }

        public Builder systemId(String systemId) {
            this.systemId = systemId;
            return this;
        }

        public Builder lineNumber(int lineNumber) {
            this.lineNumber = Math.max(0, lineNumber);
            return this;
        }

        public Builder columnNumber(int columnNumber) {
            this.columnNumber = Math.max(0, columnNumber);
            return this;
        }

        public TextFileLocation build() {
            return new TextFileLocation(systemId,
                                        lineNumber,
                                        columnNumber);
        }
    }
}