package cdc.issues;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import cdc.util.lang.Checks;

/**
 * A Diagnosis is a list of issues.
 *
 * @author Damien Carbonne
 *
 * @param <I> The issue type. <b>WARNING:</b> deprecated (since 2023-10-14). Use Issue.
 */
public final class Diagnosis<I extends Issue> {
    private final List<I> issues;

    public static final Diagnosis<?> OK_DIAGNOSIS = new Diagnosis<>();

    @SafeVarargs
    protected Diagnosis(I... issues) {
        this(Arrays.asList(Checks.isNotNull(issues, "issues")));
    }

    private Diagnosis(Collection<I> issues) {
        Checks.isNotNull(issues, "issues");

        if (issues.isEmpty()) {
            this.issues = Collections.emptyList();
        } else {
            final List<I> tmp = new ArrayList<>();
            for (final I issue : issues) {
                tmp.add(issue);
            }
            this.issues = Collections.unmodifiableList(tmp);
        }
    }

    public static <I extends Issue> Diagnosis<I> okDiagnosis() {
        @SuppressWarnings("unchecked")
        final Diagnosis<I> tmp = (Diagnosis<I>) OK_DIAGNOSIS;
        return tmp;
    }

    @Deprecated(since = "2025-01-02", forRemoval = true)
    public static <I extends Issue> Diagnosis<I> create(Collection<I> issues) {
        return of(issues);
    }

    @Deprecated(since = "2025-01-02", forRemoval = true)
    @SafeVarargs
    public static <I extends Issue> Diagnosis<I> create(I... issues) {
        return of(issues);
    }

    public static <I extends Issue> Diagnosis<I> of(Collection<I> issues) {
        if (issues.isEmpty()) {
            return okDiagnosis();
        } else {
            return new Diagnosis<>(issues);
        }
    }

    @SafeVarargs
    public static <I extends Issue> Diagnosis<I> of(I... issues) {
        return of(Arrays.asList(issues));
    }

    public List<I> getIssues() {
        return issues;
    }

    public boolean isOk() {
        return issues.isEmpty();
    }

    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder();
        if (isOk()) {
            builder.append("OK");
        } else {
            builder.append("KO ");
            for (final I issue : issues) {
                builder.append(issue);
            }
        }
        return builder.toString();
    }
}