package cdc.issues;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Utility used to make computations expected by {@link IssuesCountItem}.
 * <p>
 * If hash computation is enabled, id of issues must be stored, which may require memory.
 *
 * @author Damien Carbonne
 */
public class IssuesCounter implements IssuesCountItem {
    /** If true, ids are stored and hash is computed. */
    private final boolean computeHash;
    private int count = 0;
    private final List<IssueId> ids;
    private String hash = null;

    /**
     * @param computeHash If {@code true}, then has is computed, which means that issues ids must be stored.
     */
    public IssuesCounter(boolean computeHash) {
        this.computeHash = computeHash;
        this.ids = computeHash ? new ArrayList<>() : Collections.emptyList();
    }

    public void consume(Issue issue) {
        this.count++;
        if (computeHash) {
            this.hash = null;
            this.ids.add(issue.getId());
        }
    }

    @Override
    public int getNumberOfIssues() {
        return count;
    }

    @Override
    public String getIssuesHash() {
        if (hash == null && computeHash) {
            hash = IssueUtils.getHash(ids);
        }
        return hash;
    }
}