package cdc.issues;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import cdc.util.lang.Checks;

/**
 * Interface describing a snapshot.
 *
 * @author Damien Carbonne
 */
public interface Snapshot extends MetasItem, LabelsItem {
    /**
     * @return The project to which this snapshot belongs.
     */
    public Optional<Project> getProject();

    /**
     * @return The timestamp of this snapshot.
     */
    public Instant getTimestamp();

    /**
     * @return The name of this snapshot.
     */
    public String getName();

    /**
     * @return The description of this snapshot.
     */
    public String getDescription();

    /**
     * @return The issues of this snapshot.
     */
    public List<Issue> getIssues();

    /**
     * @return The issues hash.
     */
    public default String getIssuesHash() {
        return IssueUtils.getHash(getIssues());
    }

    public static Builder builder(Project project) {
        return new Builder(project);
    }

    public static Builder builder() {
        return new Builder(null);
    }

    public static class Builder implements MetasBuilding<Builder>, LabelsBuilding<Builder> {
        final Project project;
        String name;
        String description;
        final Metas.Builder metas = Metas.builder();
        Labels labels = Labels.NO_LABELS;
        Instant timestamp = Instant.now();
        final List<Issue> issues = new ArrayList<>();

        Builder(Project project) {
            this.project = project;
        }

        public Builder name(String name) {
            this.name = name;
            return self();
        }

        public Builder description(String description) {
            this.description = description;
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value) {
            this.metas.meta(name, value);
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value,
                            String separator) {
            this.metas.meta(name, value, separator);
            return self();
        }

        @Override
        public Builder metas(Metas metas) {
            this.metas.metas(metas);
            return self();
        }

        @Override
        public Builder labels(Labels labels) {
            this.labels = labels;
            return self();
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return self();
        }

        public Builder issue(Issue issue) {
            issues.add(Checks.isNotNull(issue, "issue"));
            return self();
        }

        public Builder issues(List<Issue> issues) {
            this.issues.addAll(issues);
            return self();
        }

        public Snapshot build() {
            return new SnapshotImpl(this);
        }
    }
}

class SnapshotImpl implements Snapshot {
    final Optional<Project> project;
    final String name;
    final String description;
    final Metas metas;
    final Labels labels;
    final Instant timestamp;
    final List<Issue> issues;
    final String issuesHash;

    public SnapshotImpl(Builder builder) {
        this.project = Optional.ofNullable(builder.project);
        this.name = builder.name;
        this.description = builder.description;
        this.metas = builder.metas.build();
        this.labels = builder.labels;
        this.timestamp = builder.timestamp;
        this.issues = List.copyOf(builder.issues);
        this.issuesHash = IssueUtils.getHash(issues);

        final Project p = project.orElse(null);
        if ((p instanceof final ProjectImpl pi)) {
            pi.snapshots.add(this);
        }
    }

    @Override
    public Optional<Project> getProject() {
        return project;
    }

    @Override
    public Instant getTimestamp() {
        return timestamp;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Metas getMetas() {
        return metas;
    }

    @Override
    public Labels getLabels() {
        return labels;
    }

    @Override
    public List<Issue> getIssues() {
        return issues;
    }

    @Override
    public String getIssuesHash() {
        return issuesHash;
    }
}