package cdc.issues.rules;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import cdc.issues.IssueSeverity;
import cdc.issues.Labels;
import cdc.issues.LabelsBuilding;
import cdc.issues.LabelsItem;
import cdc.issues.Metas;
import cdc.issues.MetasBuilding;
import cdc.issues.MetasItem;
import cdc.issues.Params;
import cdc.util.lang.Checks;
import cdc.util.lang.NotFoundException;

/**
 * Interface describing a Profile.
 * <p>
 * It is a set of {@link ConfiguredRule ConfiguredRules}, some of which are enabled.
 *
 * @author Damien Carbonne
 */
public interface Profile extends MetasItem, LabelsItem {
    /**
     * The empty profile.
     */
    public static final Profile EMPTY =
            Profile.builder()
                   .name("Empty")
                   .build();

    /**
     * @return The profile name.
     */
    public String getName();

    /**
     * @return The profile description.
     */
    public String getDescription();

    /**
     * @return The set of rules in this profile.
     */
    public Set<Rule> getRules();

    public default boolean hasRule(Rule rule) {
        return getRules().contains(rule);
    }

    /**
     * @return The set of rules that are enabled in this profile.
     */
    public default Set<Rule> getEnabledRules() {
        return getRules().stream()
                         .filter(this::isEnabled)
                         .collect(Collectors.toSet());
    }

    public ProfileConfig getProfileConfig();

    /**
     * @param ruleId The rule id.
     * @return {@code true} if a rule with {@code ruleId} exists in this profile.
     */
    public default boolean hasRule(RuleId ruleId) {
        return getProfileConfig().getRuleIds().contains(ruleId);
    }

    /**
     * @param ruleId The rule id.
     * @return The rule with {@code ruleId}.
     */
    public Optional<Rule> getRule(RuleId ruleId);

    /**
     * @param rule The rule.
     * @return The configured rule corresponding to {@code rule}.
     */
    public ConfiguredRule getConfiguredRule(Rule rule);

    /**
     * @param rule The rule.
     * @return The {@code rule} config.
     */
    public default RuleConfig getRuleConfig(Rule rule) {
        return getConfiguredRule(rule).getConfig();
    }

    /**
     * @param rule The rule.
     * @return {@code true} if {@code rule} is enabled in this profile.
     * @throws IllegalArgumentException When {@code rule} is {@code null} or is unknown.
     */
    public default boolean isEnabled(Rule rule) {
        return getConfiguredRule(rule).isEnabled();
    }

    /**
     *
     * @param rule The rule.
     * @return The optional customized severity.
     */
    public default Optional<IssueSeverity> getCustomizedSeverity(Rule rule) {
        return getConfiguredRule(rule).getCustomizedSeverity();
    }

    /**
     * @param rule The rule.
     * @return The effective severity of {@code rule}.
     * @throws IllegalArgumentException When {@code rule} is {@code null} or is unknown.
     */
    public default IssueSeverity getEffectiveSeverity(Rule rule) {
        return getConfiguredRule(rule).getEffectiveSeverity();
    }

    /**
     * Returns the parameters to use for a rule.
     *
     * @param rule The rule.
     * @return The parameters that are used to configure {@code rule}
     *         in this profile.
     * @throws IllegalArgumentException When {@code rule} is {@code null} or is unknown.
     */
    public default Params getParams(Rule rule) {
        return getConfiguredRule(rule).getParams();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder implements MetasBuilding<Builder>, LabelsBuilding<Builder> {
        String name;
        String description = "";
        final Metas.Builder metas = Metas.builder();
        Labels labels = Labels.NO_LABELS;
        final Map<RuleId, ConfiguredRule.Builder> configuredRules = new HashMap<>();

        Builder() {
        }

        public Builder profile(Profile profile) {
            name(profile.getName());
            description(profile.getDescription());
            metas(profile.getMetas());
            labels(profile.getLabels());
            for (final Rule rule : profile.getRules()) {
                rule(rule);
                enabled(rule, profile.isEnabled(rule));
                if (profile.getCustomizedSeverity(rule).isPresent()) {
                    customizedSeverity(rule, profile.getCustomizedSeverity(rule).orElseThrow());
                }
            }
            return self();
        }

        public boolean containsRule(RuleId ruleId) {
            return configuredRules.containsKey(ruleId);
        }

        public Optional<Rule> getRule(RuleId ruleId) {
            if (containsRule(ruleId)) {
                return Optional.of(configuredRules.get(ruleId).getRule());
            } else {
                return Optional.empty();
            }
        }

        public Builder name(String name) {
            this.name = name;
            return self();
        }

        public Builder description(String description) {
            this.description = description;
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value) {
            this.metas.meta(name, value);
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value,
                            String separator) {
            this.metas.meta(name, value, separator);
            return self();
        }

        @Override
        public Builder metas(Metas metas) {
            this.metas.metas(metas);
            return self();
        }

        @Override
        public Builder labels(Labels labels) {
            this.labels = labels;
            return self();
        }

        public Builder rule(Rule rule) {
            Checks.isNotNull(rule, "rule");
            Checks.assertFalse(configuredRules.containsKey(rule.getId()),
                               "Duplicate rule id {} in '{}', cannot add '{}'.",
                               rule.getId(),
                               name,
                               rule.getTitle());
            configuredRules.put(rule.getId(),
                                ConfiguredRule.builder(rule));
            return self();
        }

        public Builder enabled(Rule rule,
                               boolean enabled) {
            final ConfiguredRule.Builder b = configuredRules.computeIfAbsent(rule.getId(), k -> ConfiguredRule.builder(rule));
            b.enabled(enabled);
            return self();
        }

        public Builder enabled(RuleId ruleId,
                               boolean enabled) {
            Checks.isTrue(containsRule(ruleId), "Unknown rule {}", ruleId);

            enabled(getRule(ruleId).orElseThrow(), enabled);
            return self();
        }

        public Builder allEnabled(boolean enabled) {
            for (final ConfiguredRule.Builder b : configuredRules.values()) {
                b.enabled(enabled);
            }
            return self();
        }

        public Builder customizedSeverity(Rule rule,
                                          IssueSeverity customizedSeverity) {
            final ConfiguredRule.Builder b = configuredRules.computeIfAbsent(rule.getId(), k -> ConfiguredRule.builder(rule));
            b.customizedSeverity(customizedSeverity);
            return self();
        }

        public Builder params(Rule rule,
                              Params params) {
            final ConfiguredRule.Builder b = configuredRules.computeIfAbsent(rule.getId(), k -> ConfiguredRule.builder(rule));
            b.params(params);
            return self();
        }

        public Builder config(Rule rule,
                              RuleConfig config) {
            final ConfiguredRule.Builder b = configuredRules.computeIfAbsent(rule.getId(), k -> ConfiguredRule.builder(rule));
            b.config(config);
            return self();
        }

        public Builder config(RuleId ruleId,
                              RuleConfig config) {
            Checks.isTrue(containsRule(ruleId), "Unknown rule {}", ruleId);
            return config(getRule(ruleId).orElseThrow(), config);
        }

        public Builder config(ProfileConfig profileConfig) {
            for (final RuleId ruleId : profileConfig.getRuleIds()) {
                if (containsRule(ruleId)) {
                    final RuleConfig config = profileConfig.getRuleConfig(ruleId).orElseThrow();
                    config(ruleId, config);
                }
            }
            return self();
        }

        public ProfileImpl build() {
            return new ProfileImpl(this);
        }
    }
}

class ProfileImpl implements Profile {
    private static final String RULE = "rule";

    /** The profile name. */
    private final String name;
    /** The profile description. */
    private String description = "";
    /** The profile meta datas. */
    private Metas metas = Metas.NO_METAS;
    /** The profile labels. */
    private Labels labels = Labels.NO_LABELS;
    /** The declared rules and their associated config. */
    private final Map<RuleId, ConfiguredRule> configuredRules = new HashMap<>();

    protected ProfileImpl(Builder builder) {
        this.name = Checks.isNotNull(builder.name, "name");
        this.description = Checks.isNotNull(builder.description, "description");
        this.metas = builder.metas.build();
        this.labels = Checks.isNotNull(builder.labels, "labels");
        for (final Map.Entry<RuleId, ConfiguredRule.Builder> entry : builder.configuredRules.entrySet()) {
            configuredRules.put(entry.getKey(), entry.getValue().build());
        }
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Metas getMetas() {
        return metas;
    }

    @Override
    public Labels getLabels() {
        return labels;
    }

    @Override
    public Optional<Rule> getRule(RuleId ruleId) {
        if (configuredRules.containsKey(ruleId)) {
            return Optional.ofNullable(configuredRules.get(ruleId).getRule());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public Set<Rule> getRules() {
        return configuredRules.values()
                              .stream()
                              .map(ConfiguredRule::getRule)
                              .collect(Collectors.toSet());
    }

    @Override
    public boolean hasRule(Rule rule) {
        return rule != null && configuredRules.containsKey(rule.getId());
    }

    @Override
    public ProfileConfig getProfileConfig() {
        final ProfileConfig.Builder result = ProfileConfig.builder();
        for (final ConfiguredRule cr : configuredRules.values()) {
            result.set(cr.getRule().getId(), cr.getConfig());
        }
        return result.build();
    }

    @Override
    public boolean hasRule(RuleId ruleId) {
        return configuredRules.containsKey(ruleId);
    }

    @Override
    public ConfiguredRule getConfiguredRule(Rule rule) {
        Checks.isNotNull(rule, RULE);
        final ConfiguredRule result = configuredRules.get(rule.getId());
        if (result == null) {
            throw new NotFoundException("Unknown rule " + rule.getId());
        } else {
            return result;
        }
    }
}