package cdc.issues;

import java.util.regex.Pattern;

import cdc.util.lang.Checks;

/**
 * Definition of a formal parameter. It has:
 * <ul>
 * <li>A name
 * <li>An optional description
 * </ul>
 *
 * @author Damien Carbonne
 */
public interface FormalParam extends Comparable<FormalParam> {
    /**
     * @return The formal parameter name.
     */
    public String getName();

    /**
     * @return The formal parameter description.
     */
    public String getDescription();

    /**
     * Creates a FormalParam.
     *
     * @param name The name.
     * @param description The description.
     * @return A new FormalParam.
     * @throws IllegalArgumentException When {@code name} is not valid.
     */
    public static FormalParam of(String name,
                                 String description) {
        return new FormalParamImpl(name, description);
    }

    /**
     * Returns {@code true} if a string is a valid name:
     * <ul>
     * <li>It cannot be null
     * <li>It cannot be empty
     * <li>It cannot contain any space or control character
     * </ul>
     *
     * @param name The name.
     * @return {@code true} if {@code label} is a valid label.
     */
    public static boolean isValidName(String name) {
        return name != null
                && FormalParamImpl.VALID_NAME_PATTERN.matcher(name).matches();
    }
}

record FormalParamImpl(String name,
                       String description)
        implements FormalParam {
    /** Accept anything except spaces */
    static final Pattern VALID_NAME_PATTERN = Pattern.compile("^\\w[\\w\\-/ ]*$");

    FormalParamImpl {
        Checks.isTrue(FormalParam.isValidName(name), "name");
    }

    @Override
    public int compareTo(FormalParam other) {
        return name.compareTo(other.getName());
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String toString() {
        return "[" + name + ": " + description + "]";
    }
}