package cdc.issues;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import cdc.util.lang.Checks;

/**
 * Formal parameters.
 * <p>
 * It is a set of {@link FormalParam}s.
 * There can be at most one parameter with a given name.
 *
 * @author Damien Carbonne
 */
public interface FormalParams {
    public static final FormalParams NO_PARAMS = builder().build();

    /**
     * @return {@code true} if there are no parameters.
     */
    public boolean isEmpty();

    /**
     * @return The parameter names.
     */
    public Set<String> getNames();

    /**
     * @return The sorted parameter names.
     */
    public List<String> getSortedNames();

    /**
     * @return The sorted parameters.
     */
    public List<FormalParam> getSortedParams();

    /**
     * @param name The parameter name.
     * @return The parameter named {@code name}.
     * @throws IllegalArgumentException When {@code name} is not a valid parameter name.
     */
    public FormalParam getParam(String name);

    /**
     * @param name The parameter name.
     * @return The description of parameter named {@code name}.
     */
    public String getDescription(String name);

    /**
     * @return A new {@link Builder} of {@link FormalParams}.
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder of {@link FormalParams}.
     */
    public static final class Builder {
        private final Map<String, FormalParam> map = new HashMap<>();
        private static final String NAME = "name";

        private Builder() {
        }

        public Builder param(String name,
                             String description) {
            Checks.isNotNull(name, NAME);
            map.put(name, FormalParam.of(name, description));
            return this;
        }

        public FormalParams build() {
            return new FormalParamsImpl(map);
        }
    }
}

record FormalParamsImpl(Map<String, FormalParam> map) implements FormalParams {
    @Override
    public String toString() {
        return getSortedNames().stream()
                               .map(name -> name + ":" + getParam(name))
                               .collect(Collectors.joining(" ", "[", "]"));
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public Set<String> getNames() {
        return map.keySet();
    }

    @Override
    public List<String> getSortedNames() {
        return map.keySet()
                  .stream()
                  .sorted()
                  .toList();
    }

    @Override
    public List<FormalParam> getSortedParams() {
        return map.values()
                  .stream()
                  .sorted()
                  .toList();
    }

    @Override
    public FormalParam getParam(String name) {
        return map.get(name);
    }

    @Override
    public String getDescription(String name) {
        return getParam(name).getDescription();
    }
}