package cdc.issues;

import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.HashSet;
import java.util.Set;

import cdc.office.ss.WorkbookWriterFeatures;
import cdc.office.ss.WorkbookWriterFeatures.Feature;

public class IssuesFactoryFeatures {
    public static final IssuesFactoryFeatures UTC_BEST =
            builder().workbookWriterFeatures(WorkbookWriterFeatures.builder()
                                                                   .enable(Feature.AUTO_FILTER_COLUMNS)
                                                                   .enable(Feature.AUTO_SIZE_COLUMNS)
                                                                   .enable(Feature.TRUNCATE_CELLS)
                                                                   .build())
                     .hint(Hint.PRETTY_PRINT)
                     .hint(Hint.VERBOSE)
                     .zoneId(ZoneOffset.UTC)
                     .build();

    public static final IssuesFactoryFeatures UTC_FASTEST =
            builder().workbookWriterFeatures(WorkbookWriterFeatures.builder()
                                                                   .enable(Feature.AUTO_FILTER_COLUMNS)
                                                                   .enable(Feature.TRUNCATE_CELLS)
                                                                   .build())
                     .hint(Hint.PRETTY_PRINT)
                     .hint(Hint.VERBOSE)
                     .zoneId(ZoneOffset.UTC)
                     .build();

    private final WorkbookWriterFeatures workbookWriterFeatures;
    private final Set<Hint> hints = new HashSet<>();
    /** ZoneId used to print timestamp. */
    private final ZoneId zoneId;
    private final String password;

    private IssuesFactoryFeatures(WorkbookWriterFeatures workbookWriterFeatures,
                                  Set<Hint> hints,
                                  ZoneId zoneId,
                                  String password) {
        this.workbookWriterFeatures = workbookWriterFeatures;
        this.hints.addAll(hints);
        this.zoneId = zoneId;
        this.password = password;
    }

    public enum Hint {
        PRETTY_PRINT,
        VERBOSE
    }

    public WorkbookWriterFeatures getWorkbookWriterFeatures() {
        return workbookWriterFeatures;
    }

    public boolean isEnabled(Hint hint) {
        return hints.contains(hint);
    }

    public ZoneId getZoneId() {
        return zoneId;
    }

    public String getPassword() {
        return password;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private WorkbookWriterFeatures workbookWriterFeatures = WorkbookWriterFeatures.STANDARD_FAST;
        private final Set<Hint> hints = new HashSet<>();
        private ZoneId zoneId = ZoneOffset.UTC;
        private String password = null;

        protected Builder() {
            super();
        }

        public Builder workbookWriterFeatures(WorkbookWriterFeatures features) {
            this.workbookWriterFeatures = features;
            return this;
        }

        public Builder hint(Hint hint) {
            hints.add(hint);
            return this;
        }

        public Builder zoneId(ZoneId zoneId) {
            this.zoneId = zoneId;
            return this;
        }

        public Builder password(String password) {
            this.password = password;
            return this;
        }

        public IssuesFactoryFeatures build() {
            return new IssuesFactoryFeatures(workbookWriterFeatures,
                                             hints,
                                             zoneId,
                                             password);
        }
    }
}