package cdc.issues.rules;

import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

import cdc.issues.rules.IssuesDetector.Descriptor;
import cdc.util.lang.Checks;

/**
 * Implementation of {@link RulesCatalog} that aggregates delegates catalogs.
 * <p>
 * This implementation does not cache any data.
 *
 * @author Damien Carbonne
 */
public class CompositeRulesCatalog implements RulesCatalog {
    private final Set<RulesCatalog> catalogs = new LinkedHashSet<>();

    /**
     * Adds a new delegate catalog.
     *
     * @param catalog The catalog to add.
     * @return This catalog.
     */
    public CompositeRulesCatalog add(RulesCatalog catalog) {
        Checks.isNotNull(catalog, "catalog");

        catalogs.add(catalog);

        return this;
    }

    @Override
    public Set<String> getDomains() {
        final Set<String> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getDomains());
        }
        return set;
    }

    @Override
    public Set<Rule> getRules() {
        final Set<Rule> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getRules());
        }
        return set;
    }

    @Override
    public Set<Descriptor<?>> getDescriptors() {
        final Set<Descriptor<?>> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getDescriptors());
        }
        return set;
    }

    @Override
    public <T> Set<Descriptor<T>> getDescriptors(Class<T> dataClass) {
        final Set<Descriptor<T>> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getDescriptors(dataClass));
        }
        return set;
    }

    @Override
    public <T> Descriptor<T> getDescriptorOrNull(Rule rule,
                                                 Class<T> dataClass) {
        for (final RulesCatalog catalog : catalogs) {
            final Descriptor<T> descriptor = catalog.getDescriptorOrNull(rule, dataClass);
            if (descriptor != null) {
                return descriptor;
            }
        }
        return null;
    }
}