package cdc.issues.rules;

import java.util.Objects;

import cdc.issues.FormalParams;
import cdc.issues.IssueSeverity;
import cdc.issues.IssueSeverityItem;
import cdc.util.lang.Checks;

/**
 * Class used to describe a Rule.
 *
 * @author Damien Carbonne
 */
public class Rule {
    private final RuleId id;
    private final IssueSeverity severity;
    private final String description;
    private final FormalParams params;

    private Rule(RuleId id,
                 IssueSeverity severity,
                 String description,
                 FormalParams params) {
        this.id = Checks.isNotNull(id, "id");
        this.severity = Checks.isNotNull(severity, "severity");
        this.description = Checks.isNotNull(description, "description");
        this.params = Checks.isNotNull(params, "params");
    }

    /**
     * @return The rule identifier.
     */
    public RuleId getId() {
        return id;
    }

    /**
     * @return The rule domain.
     */
    public String getDomain() {
        return id.getDomain();
    }

    /**
     * @return The rule name.
     */
    public String getName() {
        return id.getName();
    }

    /**
     * @param <T> The enum type.
     * @param cls The enum class.
     * @return The rule name as an enum.
     */
    public <T extends Enum<T>> T getName(Class<T> cls) {
        return id.getName(cls);
    }

    /**
     * @return The rule severity.
     */
    public IssueSeverity getSeverity() {
        return severity;
    }

    /**
     * @return The rule description.
     */
    public String getDescription() {
        return description;
    }

    /**
     * @return The rule parameters.
     */
    public FormalParams getParams() {
        return params;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id,
                            severity,
                            description,
                            params);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof Rule)) {
            return false;
        }
        final Rule other = (Rule) object;
        return Objects.equals(this.id, other.id)
                && this.severity == other.severity
                && Objects.equals(this.description, other.description)
                && Objects.equals(this.params, other.params);
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * Rule builder.
     *
     * @author Damien Carbonne
     */
    public static class Builder {
        private String domain;
        private String name;
        private IssueSeverity severity;
        private String description;
        private FormalParams params = FormalParams.NO_PARAMS;

        protected Builder() {
        }

        /**
         * Sets the rule domain.
         *
         * @param domain The domain.
         * @return This builder.
         */
        public Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        /**
         * Sets the rule name.
         *
         * @param name The name.
         * @return This builder.
         */
        public Builder name(String name) {
            this.name = name;
            return this;
        }

        /**
         * Sets the rule name and its severity if the {@code name} implements
         * {@link IssueSeverityItem} and current severity is {@code null}.
         *
         * @param name The name.
         * @return This builder.
         */
        public Builder name(Enum<?> name) {
            this.name = name.name();
            if (severity == null && name instanceof IssueSeverityItem) {
                severity(((IssueSeverityItem) name).getSeverity());
            }
            return this;
        }

        /**
         * Sets the rule severity.
         *
         * @param severity The severity.
         * @return This builder.
         */
        public Builder severity(IssueSeverity severity) {
            this.severity = severity;
            return this;
        }

        /**
         * Sets the rule description.
         *
         * @param description The description.
         * @return This builder.
         */
        public Builder description(String description) {
            this.description = description;
            return this;
        }

        /**
         * Sets the rule formal parameters.
         *
         * @param params The parameters.
         * @return This builder.
         */
        public Builder params(FormalParams params) {
            this.params = params;
            return this;
        }

        public Rule build() {
            return new Rule(new RuleId(domain, name),
                            severity,
                            description,
                            params);
        }
    }
}