package cdc.issues.rules;

import java.util.HashSet;
import java.util.Set;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import cdc.issues.Issue;
import cdc.issues.IssuesHandler;
import cdc.issues.Params;
import cdc.issues.locations.LocatedData;
import cdc.util.events.ProgressController;
import cdc.util.events.ProgressSupplier;
import cdc.util.lang.Checks;
import cdc.util.lang.NotFoundException;

/**
 * Interface implemented by classes that manage and domains, rules and associated services.
 *
 * @author Damien Carbonne
 */
public interface RulesCatalog {
    /**
     * @return A set of all known domains.
     */
    public Set<String> getDomains();

    /**
     * @param domain The domain
     * @return {@code true} if {@code domain} is known.
     */
    public default boolean hasDomain(String domain) {
        return getDomains().contains(domain);
    }

    /**
     * @return A set of all known {@link Rule}s.
     */
    public Set<Rule> getRules();

    /**
     * Returns {@code true} if a rule is registered.
     *
     * @param rule The rule
     * @return {@code true} if {@code rule} is registered.
     */
    public default boolean hasRule(Rule rule) {
        return getRules().contains(rule);
    }

    /**
     * Returns a set of all known rules belonging to a domain.
     *
     * @param domain The domain.
     * @return A set of rules belonging to {@code domain}.
     */
    public default Set<Rule> getRules(String domain) {
        return getRules().stream()
                         .filter(rule -> rule.getDomain().equals(domain))
                         .collect(Collectors.toSet());
    }

    /**
     * @return A set of all known {@link IssuesDetector.Descriptor}s.
     */
    public Set<IssuesDetector.Descriptor<?>> getDescriptors();

    /**
     * Returns a set of all Descriptors registered for a data class.
     *
     * @param <T> The data type.
     * @param dataClass The data class.
     * @return A set of all Descriptors registered for {@code dataClass}.
     */
    public <T> Set<IssuesDetector.Descriptor<T>> getDescriptors(Class<T> dataClass);

    /**
     * Returns the {@link IssuesDetector.Descriptor} associated to a Rule and
     * a Class, or {@code null}.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @return The {@link IssuesDetector.Descriptor} associated to {@code rule}
     *         and {@code dataClass}, or {@code null}.
     */
    public <T> IssuesDetector.Descriptor<T> getDescriptorOrNull(Rule rule,
                                                                Class<T> dataClass);

    /**
     * Returns the {@link IssuesDetector.Descriptor} associated to a Rule and a Class.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @return The {@link IssuesDetector.Descriptor} associated to {@code rule} and {@code dataClass}.
     * @throws NotFoundException When no matching Descriptor could be found.
     */
    public default <T> IssuesDetector.Descriptor<T> getDescriptor(Rule rule,
                                                                  Class<T> dataClass) {
        final IssuesDetector.Descriptor<T> descriptor = getDescriptorOrNull(rule, dataClass);
        if (descriptor == null) {
            throw new NotFoundException("No descriptor found for [" + rule.getName() + ", " + dataClass.getCanonicalName() + "]");
        } else {
            return descriptor;
        }
    }

    /**
     * Returns {@code true} if an {@link IssuesDetector.Descriptor}
     * is registered for a Rule and a Class.
     *
     * @param rule The rule.
     * @param dataClass The data class
     * @return {@code true} if an {@link IssuesDetector.Descriptor} is registered
     *         for {@code rule} and {@code dataClass}.
     */
    public default boolean hasDescriptor(Rule rule,
                                         Class<?> dataClass) {
        return getDescriptorOrNull(rule, dataClass) != null;
    }

    /**
     * Creates and configures one IssuesDetector for one rule and parameters.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @param params The detector effective parameters.
     * @return An IssueDetector matching {@code rule} and {@code dataClass} and configured with {@code params}.
     * @throws NotFoundException When no matching IssuesDetector could be found.
     */
    public default <T> IssuesDetector<T> createIssuesDetector(Rule rule,
                                                              Class<T> dataClass,
                                                              Params params) {
        final IssuesDetector.Descriptor<T> descriptor = getDescriptor(rule, dataClass);
        final Set<Rule> rules = new HashSet<>();
        rules.add(rule);
        return descriptor.create(params, rules);
    }

    public default <T> IssuesDetector<T> createIssuesDetector(Rule rule,
                                                              Class<T> dataClass) {
        return createIssuesDetector(rule, dataClass, Params.NO_PARAMS);
    }

    /**
     * Creates and configures an appropriate IssuesDetector, and applies it to data provided by a Spliterator.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @param params The rule effective parameters.
     * @param spliterator The data spliterator.
     * @param issuesHandler The issues handler that will collect issues.
     * @param controller The progress controller.
     */
    public default <T> void apply(Rule rule,
                                  Class<T> dataClass,
                                  Params params,
                                  Spliterator<LocatedData<T>> spliterator,
                                  IssuesHandler<Issue> issuesHandler,
                                  ProgressController controller) {
        final IssuesDetector<T> detector = createIssuesDetector(rule, dataClass, params);

        apply(detector, spliterator, issuesHandler, controller);
    }

    /**
     * Create all possible IssueDectectors for a data class, configures them with all supported rules,
     * and apply them to data provided by a Spliterator.
     * <p>
     * <b>Note:</b> The passed params must be usable for all created detectors,
     * each using the appropriate parameters and ignoring those that are meaningless.
     *
     * @param <T> The data type.
     * @param dataClass The data class.
     * @param params The rule effective parameters.
     * @param spliterator The data spliterator.
     * @param issuesHandler The issues handler that will collect issues.
     * @param controller The progress controller.
     */
    public default <T> void apply(Class<T> dataClass,
                                  Params params,
                                  Spliterator<LocatedData<T>> spliterator,
                                  IssuesHandler<Issue> issuesHandler,
                                  ProgressController controller) {
        for (final IssuesDetector.Descriptor<T> descriptor : getDescriptors(dataClass)) {
            final IssuesDetector<T> detector = descriptor.create(params, descriptor.getRules());
            apply(detector, spliterator, issuesHandler, controller);
        }
    }

    /**
     * Applies an IssuesDetector to data provided by a Spliterator.
     *
     * @param <T> The data type.
     * @param detector The detector.
     * @param spliterator The data spliterator.
     * @param issuesHandler The issues handler that will collect issues.
     * @param controller The progress controller.
     */
    public static <T> void apply(IssuesDetector<T> detector,
                                 Spliterator<LocatedData<T>> spliterator,
                                 IssuesHandler<Issue> issuesHandler,
                                 ProgressController controller) {
        Checks.isNotNull(detector, "detector");
        Checks.isNotNull(spliterator, "spliterator");
        Checks.isNotNull(issuesHandler, "issuesHandler");
        Checks.isNotNull(controller, "controller");

        final ProgressSupplier progress = new ProgressSupplier(controller);
        progress.reset(spliterator.estimateSize(), IssuesDetector.toString(detector));

        final Consumer<LocatedData<T>> consumer = ld -> {
            detector.analyze(ld.getData(), ld.getLocation(), issuesHandler);
        };

        boolean next = true;
        while (next) {
            next = next && spliterator.tryAdvance(consumer);
            next = next && !controller.isCancelled();
            progress.incrementValue();
        }
    }
}