package cdc.issues.impl;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertSame;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.jupiter.api.Test;

import cdc.issues.Issue;
import cdc.issues.IssueSeverity;
import cdc.issues.IssuesCollector;
import cdc.issues.IssuesHandler;
import cdc.issues.VerboseIssuesHandler;
import cdc.issues.locations.DefaultLocatedData;
import cdc.issues.locations.DefaultLocation;
import cdc.issues.locations.LocatedData;
import cdc.issues.locations.Location;
import cdc.issues.rules.AbstractIssueDetector;
import cdc.issues.rules.CompositeRulesCatalog;
import cdc.issues.rules.ConfiguredRule;
import cdc.issues.rules.DataSource;
import cdc.issues.rules.IssuesDetector;
import cdc.issues.rules.Profile;
import cdc.issues.rules.Rule;
import cdc.issues.rules.RulesCatalog;
import cdc.util.events.ProgressController;

class RulesCatalogImplTest {
    private static final Logger LOGGER = LogManager.getLogger(RulesCatalogImplTest.class);
    static final String PROJECT = "project";
    static final String SNAPSHOT = "snapshot";

    static final Rule RULE1 = Rule.builder()
                                  .domain("Domain")
                                  .name("Rule1")
                                  .addSeverity(IssueSeverity.CRITICAL)
                                  .description("Rule 1 description")
                                  .build();
    static final Rule RULE2 = Rule.builder()
                                  .domain("Domain")
                                  .name("Rule2")
                                  .addSeverity(IssueSeverity.CRITICAL)
                                  .description("Rule 2 description")
                                  .build();
    static final Rule RULE3 = Rule.builder()
                                  .domain("Domain")
                                  .name("Rule3")
                                  .addSeverity(IssueSeverity.CRITICAL)
                                  .description("Rule 3 description")
                                  .build();

    static final IssuesDetector.Factory<String> DESCRIPTOR12 =
            new IssuesDetector.AbstractFactory<String>(String.class,
                                                       RULE1,
                                                       RULE2) {
                @Override
                public IssuesDetector<String> create(String project,
                                                     String snapshot,
                                                     Set<ConfiguredRule> configuredRules) {
                    return new Detector1(project, snapshot, configuredRules);
                }
            };

    static class Detector1 extends AbstractIssueDetector<String> {
        Detector1(String project,
                  String snapshot,
                  Set<ConfiguredRule> configuredRules) {
            super(DESCRIPTOR12,
                  project,
                  snapshot,
                  configuredRules);
        }

        @Override
        public void analyze(String data,
                            List<Location> locations,
                            IssuesHandler<Issue> issuesHandler) {
            LOGGER.info("analyze({}, {})", data, locations);
            for (final ConfiguredRule crule : getEnabledConfiguredRules()) {
                issuesHandler.issue(Issue.builder()
                                         .domain(crule.getRule().getDomain())
                                         .name(crule.getRule().getName())
                                         .description("Analyzed " + data)
                                         .severity(IssueSeverity.INFO)
                                         .locations(locations)
                                         .build());
            }
        }
    }

    @Test
    void test() {
        final RulesCatalog catalog1 = new RulesCatalogImpl().register(RULE1)
                                                            .register(RULE2)
                                                            .register(RULE3)
                                                            .register(DESCRIPTOR12);

        final Profile profile1 = new ProfileImpl("profile").add(RULE1)
                                                           .add(RULE2)
                                                           .add(RULE3);

        final RulesCatalog ccatalog = new CompositeRulesCatalog().add(catalog1);

        assertEquals(1, catalog1.getDomains().size());
        assertEquals(3, catalog1.getRules().size());
        assertEquals(3, catalog1.getRules("Domain").size());
        assertEquals(0, catalog1.getRules("domain").size());
        assertEquals(1, catalog1.getFactories().size());
        assertEquals(1, catalog1.getFactories(String.class).size());
        assertTrue(catalog1.hasDescriptor(RULE1, String.class));
        assertTrue(catalog1.hasDescriptor(RULE2, String.class));
        assertFalse(catalog1.hasDescriptor(RULE3, String.class));
        assertTrue(catalog1.hasDomain("Domain"));
        assertFalse(catalog1.hasDomain("domain"));
        assertTrue(catalog1.hasRule(RULE1));

        assertEquals(String.class, DESCRIPTOR12.getDataClass());

        final IssuesDetector<String> detector1 =
                catalog1.createIssuesDetector(null,
                                              null,
                                              new ConfiguredRule(RULE1),
                                              String.class);
        assertEquals(String.class, detector1.getDataClass());
        assertSame(DESCRIPTOR12, detector1.getFactory());
        assertEquals(1, detector1.getEnabledConfiguredRules().size());
        assertTrue(detector1.getEnabledRules().contains(RULE1));

        assertThrows(IllegalArgumentException.class,
                     () -> {
                         catalog1.createIssuesDetector(null,
                                                       null,
                                                       new ConfiguredRule(RULE3),
                                                       String.class);
                     });

        assertEquals(1, ccatalog.getDomains().size());
        assertEquals(3, ccatalog.getRules().size());
        assertEquals(3, ccatalog.getRules("Domain").size());
        assertEquals(0, ccatalog.getRules("domain").size());
        assertEquals(1, ccatalog.getFactories().size());
        assertEquals(1, ccatalog.getFactories(String.class).size());
        assertTrue(ccatalog.hasDescriptor(RULE1, String.class));
        assertTrue(ccatalog.hasDescriptor(RULE2, String.class));
        assertFalse(ccatalog.hasDescriptor(RULE3, String.class));

        detector1.toString(); // Code coverage

        final IssuesDetector<String> detector12 =
                catalog1.getFactory(RULE1, String.class)
                        .get()
                        .create(PROJECT,
                                SNAPSHOT,
                                new ConfiguredRule(RULE1),
                                new ConfiguredRule(RULE2));
        detector12.toString(); // Code coverage

        assertThrows(IllegalArgumentException.class, () -> {
            DESCRIPTOR12.create(PROJECT,
                                SNAPSHOT,
                                new ConfiguredRule(RULE3));
        });

        final IssuesCollector<Issue> collector = new IssuesCollector<Issue>();
        assertSame(IssuesHandler.VOID, collector.getDelegate());
        final DefaultLocation location = new DefaultLocation("Test");

        detector1.analyze("Hello", location, collector);
        assertEquals(1, collector.getIssues().size());
        assertEquals(1, collector.getIssues(IssueSeverity.INFO).size());

        collector.clear();
        final List<LocatedData<String>> data = new ArrayList<>();
        Collections.addAll(data,
                           new DefaultLocatedData<>("Hello", location),
                           new DefaultLocatedData<>("World", location));

        catalog1.apply(PROJECT,
                       SNAPSHOT,
                       profile1,
                       new DataSource<String>(String.class,
                                              data.spliterator()),
                       new VerboseIssuesHandler<Issue>(collector),
                       ProgressController.VERBOSE);
        assertEquals(4, collector.getIssues().size());
    }
}