package cdc.issues.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import cdc.issues.IssueSeverity;
import cdc.issues.Params;
import cdc.issues.rules.ConfiguredRule;
import cdc.issues.rules.Profile;
import cdc.issues.rules.ProfileConfig;
import cdc.issues.rules.Rule;
import cdc.issues.rules.RuleConfig;
import cdc.issues.rules.RuleId;
import cdc.util.lang.Checks;
import cdc.util.lang.NotFoundException;

/**
 * Default implementation of {@link Profile}.
 *
 * @author Damien Carbonne
 */
public class ProfileImpl implements Profile {
    private static final String DESCRIPTION = "description";
    private static final String METAS = "metas";
    private static final String PARAMS = "params";
    private static final String RULE = "rule";

    /** The profile name. */
    private String name;
    /** The profile description. */
    private String description = "";
    /** The profile meta datas. */
    private Params metas = Params.NO_PARAMS;
    /** The declared rules and their associated config. */
    private final Map<RuleId, ConfiguredRule> rules = new HashMap<>();

    public ProfileImpl(String name) {
        this.name = name;
    }

    public ProfileImpl setName(String name) {
        this.name = name;
        return this;
    }

    public ProfileImpl setDescription(String description) {
        this.description = Checks.isNotNull(description, DESCRIPTION);
        return this;
    }

    public ProfileImpl setMetas(Params metas) {
        this.metas = Checks.isNotNull(metas, METAS);
        return this;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Params getMetas() {
        return metas;
    }

    public Rule getRule(RuleId ruleId) {
        return rules.get(ruleId).getRule();
    }

    /**
     * Adds a rule if it is not already added.
     *
     * @param rule The rule.
     * @return This profile.
     */
    public ProfileImpl add(Rule rule) {
        Checks.isNotNull(rule, RULE);

        if (!rules.containsKey(rule.getId())) {
            rules.put(rule.getId(),
                      ConfiguredRule.builder(rule)
                                    .enabled(true)
                                    .build());
        }
        return this;
    }

    public ProfileImpl setConfig(Rule rule,
                                 RuleConfig config) {
        add(rule);

        rules.put(rule.getId(),
                  ConfiguredRule.builder(rule)
                                .config(config)
                                .build());

        return this;
    }

    public ProfileImpl setConfig(RuleId ruleId,
                                 RuleConfig config) {
        return setConfig(getRule(ruleId), config);
    }

    /**
     * Removes a rule.
     *
     * @param rule The rule.
     * @return This object.
     */
    public ProfileImpl remove(Rule rule) {
        Checks.isNotNull(rule, RULE);

        rules.remove(rule.getId());
        return this;
    }

    public ProfileImpl setAllEnabled(boolean enabled) {
        for (final Rule rule : getRules()) {
            setEnabled(rule, enabled);
        }
        return this;
    }

    /**
     * Enables or disables a rule.
     * <p>
     * If the rule is not declared, declares it.
     *
     * @param rule The rule.
     * @param enabled {@code true} if the rule must be enabled.
     * @return This object.
     */
    public ProfileImpl setEnabled(Rule rule,
                                  boolean enabled) {
        add(rule);

        rules.put(rule.getId(),
                  ConfiguredRule.builder(rule)
                                .config(rules.get(rule.getId()).getConfig())
                                .enabled(enabled)
                                .build());
        return this;
    }

    public ProfileImpl setEnabled(RuleId ruleId,
                                  boolean enabled) {
        return setEnabled(getRule(ruleId), enabled);
    }

    public ProfileImpl setCustomizedSeverity(Rule rule,
                                             IssueSeverity customizedSeverity) {
        add(rule);

        rules.put(rule.getId(),
                  ConfiguredRule.builder(rule)
                                .config(rules.get(rule.getId()).getConfig())
                                .customizedSeverity(customizedSeverity)
                                .build());
        return this;
    }

    public ProfileImpl setCustomizedSeverity(RuleId ruleId,
                                             IssueSeverity customizedSeverity) {
        return setCustomizedSeverity(getRule(ruleId), customizedSeverity);
    }

    public ProfileImpl setParams(Rule rule,
                                 Params params) {
        Checks.isNotNull(params, PARAMS);
        add(rule);

        rules.put(rule.getId(),
                  ConfiguredRule.builder(rule)
                                .config(rules.get(rule.getId()).getConfig())
                                .params(params)
                                .build());
        return this;
    }

    public ProfileImpl setParams(RuleId ruleId,
                                 Params params) {
        return setParams(getRule(ruleId), params);
    }

    public ProfileImpl apply(ProfileConfig profileConfig) {
        for (final RuleId ruleId : profileConfig.getRuleIds()) {
            if (hasRule(ruleId)) {
                final RuleConfig config = profileConfig.getRuleConfig(ruleId);
                setConfig(ruleId, config);
            }
        }
        return this;
    }

    @Override
    public Set<Rule> getRules() {
        return rules.values()
                    .stream()
                    .map(ConfiguredRule::getRule)
                    .collect(Collectors.toSet());
    }

    @Override
    public boolean hasRule(Rule rule) {
        return rule != null && rules.containsKey(rule.getId());
    }

    @Override
    public ProfileConfig getProfileConfig() {
        final ProfileConfigImpl result = new ProfileConfigImpl();
        for (final ConfiguredRule cr : rules.values()) {
            result.set(cr.getRule().getId(), cr.getConfig());
        }
        return result;
    }

    @Override
    public boolean hasRule(RuleId ruleId) {
        return rules.containsKey(ruleId);
    }

    @Override
    public ConfiguredRule getConfiguredRule(Rule rule) {
        Checks.isNotNull(rule, RULE);
        final ConfiguredRule result = rules.get(rule.getId());
        if (result == null) {
            throw new NotFoundException("Unknown rule " + rule.getId());
        } else {
            return result;
        }
    }
}