/**
 * Copyright (C) 2007-2018 Tatsuo Satoh <multisqllib@gmail.com>
 *
 * This file is part of sqlapp-command.
 *
 * sqlapp-command is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sqlapp-command is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with sqlapp-command.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sqlapp.data.db.command;

import java.io.File;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collections;
import java.util.List;

import com.sqlapp.data.db.dialect.Dialect;
import com.sqlapp.data.db.dialect.util.SqlSplitter;
import com.sqlapp.data.db.dialect.util.SqlSplitter.SplitResult;
import com.sqlapp.data.parameter.ParametersContext;
import com.sqlapp.jdbc.sql.JdbcHandler;
import com.sqlapp.jdbc.sql.SqlConverter;
import com.sqlapp.jdbc.sql.node.SqlNode;
import com.sqlapp.util.CommonUtils;
import com.sqlapp.util.FileUtils;

public class SqlExecuteCommand extends AbstractSqlCommand{

	private List<File> sqlFiles=null;

	private List<String> sqlText=null;

	@Override
	protected void doRun() {
		Connection connection=null;
		try{
			connection=this.getConnection();
			connection.setAutoCommit(false);
			Dialect dialect=this.getDialect(connection);
			SqlSplitter sqlSplitter=dialect.createSqlSplitter();
			SqlConverter sqlConverter=getSqlConverter();
			for(File file:getSqlFiles()){
				ParametersContext context=new ParametersContext();
				context.putAll(this.getContext());
				String text=FileUtils.readText(file, this.getEncoding());
				List<SplitResult> sqls=sqlSplitter.parse(text);
				for(SplitResult splitResult:sqls){
					SqlNode sqlNode=sqlConverter.parseSql(context, splitResult.getText());
					JdbcHandler jdbcHandler=dialect.createJdbcHandler(sqlNode);
					jdbcHandler.execute(connection, context);
				}
			}
			for(String text:getSqlText()){
				ParametersContext context=new ParametersContext();
				context.putAll(this.getContext());
				List<SplitResult> sqls=sqlSplitter.parse(text);
				for(SplitResult splitResult:sqls){
					SqlNode sqlNode=sqlConverter.parseSql(context, splitResult.getText());
					JdbcHandler jdbcHandler=dialect.createJdbcHandler(sqlNode);
					jdbcHandler.execute(connection, context);
				}
			}
			connection.commit();
		} catch (SQLException e) {
			if (connection!=null){
				try {
					connection.rollback();
				} catch (SQLException e1) {
				}
			}
			this.getExceptionHandler().handle(e);
		}
	}

	/**
	 * @return the sqls
	 */
	protected List<File> getSqlFiles() {
		return sqlFiles;
	}

	/**
	 * @param createSql the sqls to set
	 */
	public void setSqlFiles(List<File> sqlFiles) {
		this.sqlFiles = sqlFiles;
	}

	/**
	 * @param createSql the sqls to set
	 */
	public void setSqlFiles(File... sqlFiles) {
		if (CommonUtils.isEmpty(sqlFiles)){
			this.sqlFiles = Collections.emptyList();
		} else{
			this.sqlFiles = CommonUtils.list(sqlFiles);
		}
	}

	/**
	 * @param sqlFiles the sqls to set
	 */
	public void setSqlFiles(String... sqlPaths) {
		if (CommonUtils.isEmpty(sqlPaths)){
			this.sqlFiles = Collections.emptyList();
		} else{
			this.sqlFiles = CommonUtils.list();
			for(String sqlPath:sqlPaths){
				this.sqlFiles.add(new File(sqlPath));
			}
		}
	}

	/**
	 * @return the sqlText
	 */
	protected List<String> getSqlText() {
		return sqlText;
	}

	/**
	 * @param sqlText the sqlText to set
	 */
	public void setSqlText(List<String> sqlText) {
		this.sqlText = sqlText;
	}
	
	/**
	 * @param sqlFiles the sqls to set
	 */
	public void setSqlText(String... args) {
		if (CommonUtils.isEmpty(args)){
			this.sqlText = Collections.emptyList();
		} else{
			this.sqlText = CommonUtils.list();
			for(String arg:args){
				if (!CommonUtils.isEmpty(arg)){
					this.sqlText.add(arg);
				}
			}
		}
	}

}
