/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package com.stackone.stackone_client_java.utils.pagination;

import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import com.jayway.jsonpath.ReadContext;
import com.jayway.jsonpath.spi.json.JacksonJsonProvider;
import com.jayway.jsonpath.spi.mapper.JacksonMappingProvider;
import com.stackone.stackone_client_java.utils.CopiableInputStream;
import com.stackone.stackone_client_java.utils.ResponseWithBody;
import com.stackone.stackone_client_java.utils.SpeakeasyLogger;

import java.io.IOException;
import java.io.InputStream;
import java.net.http.HttpResponse;
import java.util.Iterator;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;

// Internal API only

/**
 * A generic pagination implementation that handles fetching and iterating through paginated data.
 * This class implements the Iterator interface and can work with any request and response types.
 * It uses a ProgressTrackerStrategy to process pagination metadata from responses and determine
 * when to stop pagination.
 * <p>
 * The pagination flow works as follows:
 * 1. hasNext() checks if there's a current response or tries to fetch the next page
 * 2. fetchNext() builds the request using the output handler's current value
 * 3. The response is processed by the progress tracker to update pagination state
 * 4. next() returns the current response and clears it for the next iteration
 *
 * @param <ReqT>           The type of the request object
 * @param <ProgressParamT> The type of the progression parameter (e.g., page number, offset, cursor)
 */
public class Paginator<ReqT, ProgressParamT> implements Iterator<HttpResponse<InputStream>> {

    private static final SpeakeasyLogger logger = SpeakeasyLogger.getLogger(Paginator.class);

    /**
     * The initial request containing pagination parameters.
     */
    private final ReqT initialRequest;
    /**
     * The handler used to process pagination logic.
     */
    private final ProgressTrackerStrategy<ProgressParamT> progressTracker;
    /**
     * Function that sets the value for pagination.
     */
    private final BiFunction<ReqT, ProgressParamT, ReqT> requestModifier;
    /**
     * Function that fetches the next page of data.
     */
    private final Function<ReqT, HttpResponse<InputStream>> dataFetcher;

    private static final Configuration JSON_PATH_CONFIG = Configuration.defaultConfiguration()
            .jsonProvider(new JacksonJsonProvider())
            .mappingProvider(new JacksonMappingProvider())
            .addOptions(Option.SUPPRESS_EXCEPTIONS);
    private PaginationState state = PaginationState.INITIAL;
    private HttpResponse<InputStream> currentResponse = null;

    /**
     * Creates a new Paginator instance.
     *
     * @param initialRequest  The initial request to use for the first page
     * @param progressTracker The handler that processes pagination metadata from responses
     * @param requestModifier Function that sets the pagination value in the request
     * @param dataFetcher     Function that fetches the response for a given request
     */
    public Paginator(ReqT initialRequest,
                     ProgressTrackerStrategy<ProgressParamT> progressTracker,
                     BiFunction<ReqT, ProgressParamT, ReqT> requestModifier,
                     Function<ReqT, HttpResponse<InputStream>> dataFetcher) {
        this.initialRequest = initialRequest;
        this.progressTracker = progressTracker;
        this.requestModifier = requestModifier;
        this.dataFetcher = dataFetcher;
    }

    /**
     * Returns the current response if one exists.
     * This is useful for peeking at the current response without consuming it.
     *
     * @return An Optional containing the current response, or empty if none exists
     */
    public Optional<HttpResponse<InputStream>> currentResponse() {
        return Optional.ofNullable(currentResponse);
    }

    /**
     * Checks if there are more pages to fetch.
     * If there's no current response, it attempts to fetch the next page.
     * The output handler determines if there are more pages based on the response metadata.
     *
     * @return true if there are more pages to fetch, false otherwise
     */
    @Override
    public boolean hasNext() {
        if (currentResponse != null) {
            return true;
        }

        if (state != PaginationState.EXHAUSTED) {
            fetchNext();
        }

        return currentResponse != null;
    }

    /**
     * Fetches the next page of data.
     * This method:
     * 1. Builds the appropriate request using the output handler's current value
     * 2. Fetches the data using the provided fetcher function
     * 3. Parses the response using JsonPath
     * 4. Updates output tracker state
     * 5. Updates pagination state based on whether more pages are available
     *
     * @throws RuntimeException if there's an error fetching or parsing the response
     */
    private void fetchNext() {
        ProgressParamT currentValue = progressTracker.getPosition();
        ReqT request = state == PaginationState.INITIAL ?
                initialRequest :
                requestModifier.apply(initialRequest, currentValue);
        
        if (state == PaginationState.INITIAL) {
            logger.debug("Fetching initial page");
        } else {
            logger.debug("Fetching next page with position: {}", currentValue);
        }
        
        HttpResponse<InputStream> response = dataFetcher.apply(request);
        try (InputStream body = response.body()) {
            CopiableInputStream copiableInputStream = new CopiableInputStream(body);
            ReadContext respJson = JsonPath.using(JSON_PATH_CONFIG).parse(copiableInputStream.copy());
            currentResponse = new ResponseWithBody<>(response, given -> copiableInputStream.copy());
            boolean hasMorePages = progressTracker.advance(respJson);
            state = hasMorePages ? PaginationState.HAS_MORE_PAGES : PaginationState.EXHAUSTED;
            
            if (logger.isTraceEnabled()) {
                logger.trace("Page fetched - status: {}, hasMorePages: {}", response.statusCode(), hasMorePages);
            }
        } catch (IOException e) {
            logger.debug("Error fetching page: {}", e.getMessage());
            throw new RuntimeException(e);
        }
    }

    /**
     * Returns the next page of data.
     * This method returns the current response and clears it,
     * allowing the next call to hasNext() to fetch the next page if needed.
     *
     * @return The next page of data
     * @throws IllegalStateException if there is no current response available
     */
    @Override
    public HttpResponse<InputStream> next() {
        HttpResponse<InputStream> response = currentResponse()
                .orElseThrow(() -> new IllegalStateException("No more pages available"));
        currentResponse = null;
        return response;
    }

    /**
     * Enum representing the current state of pagination.
     */
    private enum PaginationState {
        /**
         * Initial state - no requests made yet
         */
        INITIAL,
        /**
         * More pages are available
         */
        HAS_MORE_PAGES,
        /**
         * No more pages available
         */
        EXHAUSTED
    }
}
